#!/usr/bin/env node
/* Gera ícones do app a partir do logo SVG remoto. */
const fs = require('fs');
const path = require('path');
const sharp = require('sharp');

const LOGO_URL =
  process.env.LOGO_URL || 'https://extratorpecesus.blob.core.windows.net/releases/logo-pec.png';

async function fetchSvg(url) {
  const res = await fetch(url);
  if (!res.ok) {
    throw new Error(`Falha ao baixar logo (HTTP ${res.status})`);
  }
  const buf = Buffer.from(await res.arrayBuffer());
  return buf;
}

function resolveFilenameFromUrl(url) {
  try {
    const u = new URL(url);
    const name = path.basename(u.pathname || '') || 'logo-source';
    return name;
  } catch (err) {
    return 'logo-source';
  }
}

async function ensureDir(dir) {
  await fs.promises.mkdir(dir, { recursive: true });
}

async function main() {
  const buildDir = path.join(__dirname, '..', 'build');
  await ensureDir(buildDir);

  const svgBuffer = await fetchSvg(LOGO_URL);
  const sourceName = resolveFilenameFromUrl(LOGO_URL);
  const sourcePath = path.join(buildDir, sourceName);
  await fs.promises.writeFile(sourcePath, svgBuffer);

  const sizes = [1024, 512, 256];
  for (const size of sizes) {
    const outPath = path.join(buildDir, `icon-${size}.png`);
    await sharp(svgBuffer)
      .resize(size, size, { fit: 'contain', background: { r: 0, g: 0, b: 0, alpha: 0 } })
      .png()
      .toFile(outPath);
  }

  // Ícone principal (electron-builder usa build/icon.png como default).
  await fs.promises.copyFile(path.join(buildDir, 'icon-1024.png'), path.join(buildDir, 'icon.png'));

  console.log('Ícones gerados em build/: icon.png, icon-1024.png, icon-512.png, icon-256.png');
  console.log(`Logo base salvo em ${sourcePath}`);
}

main().catch((err) => {
  console.error(err.message);
  process.exit(1);
});
