const { getPendingItems, logModuleEvent, setSyncResponse, markQueued } = require('../../app/core/storage');
const MODULE_ID = 'filhos_de_minas';
const FALLBACK_API_URL = (() => {
  const env = String(process.env.FILHOS_DE_MINAS_API_URL || '').trim();
  return env || 'https://filhosdeminas.saude.mg.gov.br/api';
})();
const DEBUG_DT_LOG = process.env.FILHOS_DE_MINAS_LOG_DT === 'true';

function sanitizeDocument(value) {
  if (!value && value !== 0) return null;
  return String(value).replace(/\D/g, '');
}

function formatDate(value) {
  if (!value && value !== 0) return null;
  const s = String(value).trim();
  // If already in ISO-like or yyyy-mm-dd with extra, take first 10 chars
  if (s.length >= 10) {
    const candidate = s.slice(0, 10);
    // basic check for yyyy-mm-dd or dd/mm/yyyy
    if (/^\d{4}-\d{2}-\d{2}$/.test(candidate)) return candidate;
    if (/^\d{2}\/\d{2}\/\d{4}$/.test(candidate)) {
      // convert dd/mm/yyyy -> yyyy-mm-dd
      const [d, m, y] = candidate.split('/');
      return `${y}-${m}-${d}`;
    }
  }
  // try Date parse as fallback
  const dt = new Date(s);
  if (!isNaN(dt.getTime())) return dt.toISOString().split('T')[0];
  return null;
}

function buildGroupedPayloads(rows, batchId) {
  const groups = new Map();
  for (const row of rows) {
    const sanitizedCpf = sanitizeDocument(row.cpf);
    const sanitizedCns = sanitizeDocument(row.cns);
    const sanitizedCpfOuCns = sanitizeDocument(row.cpf_ou_cns);
    const docKey = sanitizedCpf || sanitizedCns || sanitizedCpfOuCns || null;

    const key = row.co_fat_cidadao
      ? `cid:${row.co_fat_cidadao}`
      : docKey
      ? `doc:${docKey}`
      : `item:${row.item_id}`;

    if (!groups.has(key)) {
      const dumValue = formatDate(row.dum) || null;
      groups.set(key, {
        payload: {
          batch_id: batchId,
          // keep the first seen item_id for traceability
          item_id: String(row.item_id),
          gestante: {
            nome: row.nome || '—',
            cpf: sanitizedCpf || (row.doc_tipo === 'cpf' ? sanitizedCpfOuCns : null),
            cns: sanitizedCns || (row.doc_tipo === 'cns' ? sanitizedCpfOuCns : null),
            data_nascimento: formatDate(row.birth_date) || null,
            nome_mae: row.mother_name || null
          },
          gravidez: {
            dum: dumValue,
            qtd_filhos: Number(row.qtd_filhos) || 1,
            cbo: row.cbo || null,
            cid: row.cid || null,
            ciap: row.ciap || null
          },
          consultas: []
        },
        rowIds: []
      });
    }

    const group = groups.get(key);
    const dtInicialRaw = row.dt_inicial_atendimento;
    const dtInicialFormatted = formatDate(dtInicialRaw) || null;
    const consultaDate =
      dtInicialFormatted ||
      formatDate(row.data_consulta) ||
      formatDate(row.dum) ||
      new Date().toISOString().split('T')[0];
    if (DEBUG_DT_LOG) {
      const rowId =
        row.co_seq_fat_atd_ind || row.item_id || row.consulta_id || row.consultaId || 'unknown';
      console.info(
        `[Filhos de Minas] row=${rowId} dt_inicial_atendimento raw=${
          dtInicialRaw ?? 'null'
        } parsed=${dtInicialFormatted ?? 'null'}`
      );
    }
    const consultaId = row.co_seq_fat_atd_ind || row.consulta_id || row.item_id || null;

    // avoid adding duplicated consultas (same id + date)
    const alreadyHasConsulta = group.payload.consultas.some(
      (c) => (c.consulta_id === consultaId || (!c.consulta_id && !consultaId && c.data_consulta === consultaDate)) &&
             c.data_consulta === consultaDate
    );
    if (!alreadyHasConsulta) {
      group.payload.consultas.push({
        consulta_id: consultaId,
        data_consulta: consultaDate,
        profissional: row.professional_name || null
      });
    }

    // keep unique rowIds
    const rowIdNum = Number(row.item_id);
    if (!group.rowIds.includes(rowIdNum)) {
      group.rowIds.push(rowIdNum);
    }
  }

  return Array.from(groups.values());
}

function buildAttemptId() {
  return `filhos-de-minas:${Date.now()}-${Math.random().toString(36).slice(2, 8)}`;
}

function validatePayload(payload) {
  const errors = [];
  if (!payload.item_id) errors.push('item_id ausente');
  const gestante = payload.gestante || {};
  if (!gestante.nome) errors.push('gestante.nome ausente');
  if (!gestante.cpf) errors.push('gestante.cpf ausente');
  if (!gestante.data_nascimento) errors.push('gestante.data_nascimento ausente');

  const consultas = Array.isArray(payload.consultas) ? payload.consultas : [];
  if (consultas.length === 0) {
    errors.push('consultas ausente');
  } else if (!consultas[0].data_consulta) {
    errors.push('consultas[0].data_consulta ausente');
  }
  return errors;
}

async function sendRequest(url, headersObj, payloadObj) {
  const resp = await fetch(url, {
    method: 'POST',
    headers: headersObj,
    body: JSON.stringify(payloadObj)
  });
  const text = await resp.text().catch(() => null);
  return { resp, text };
}

function safeLogModuleEvent(evt) {
  try {
    logModuleEvent(evt);
  } catch (e) {
    console.error('[Filhos de Minas] Falha ao registrar log de evento', e);
  }
}

function buildCurlCommand(url, headersObj = {}, payload = null, method = 'POST') {
  const parts = ['curl', '-X', method.toUpperCase()];
  Object.entries(headersObj || {}).forEach(([header, value]) => {
    if (!value) return;
    const headerValue = `${header}: ${value}`;
    parts.push('-H', `'${headerValue}'`);
  });
  const payloadString =
    payload === null || payload === undefined
      ? ''
      : typeof payload === 'string'
      ? payload
      : JSON.stringify(payload);
  if (payloadString && payloadString !== '{}') {
    parts.push('--data', `'${payloadString.replace(/'/g, "\\'")}'`);
  }
  parts.push(`'${url}'`);
  return parts.join(' ');
}

function describeErrorDetails(err) {
  if (!err) return null;
  const parts = [];
  if (typeof err === 'object' && err.curl) {
    parts.push(err.curl);
  }
  if (err.stack) {
    parts.push(err.stack);
  } else if (typeof err === 'string') {
    parts.push(err);
  } else if (err.message) {
    parts.push(err.message);
  }
  if (parts.length === 0) return null;
  return parts.join('\n\n');
}

function delay(ms) {
  return new Promise((resolve) => setTimeout(resolve, ms));
}

function safeInitialLogs(attemptId, total) {
  try {
    safeLogModuleEvent({
      moduleId: MODULE_ID,
      attemptId,
      action: 'batch.start',
      status: 'info',
      message: `Preparando batch de ${total} registros`
    });
    safeLogModuleEvent({
      moduleId: MODULE_ID,
      attemptId,
      action: 'batch.count',
      status: 'info',
      message: `Itens encontrados para sincronização: ${total}`
    });
  } catch (e) {
    console.error('[Filhos de Minas] Falha ao registrar initial logs', e);
  }
}

async function getFilhosToken(config, baseUrl) {
  const username =
    String(config?.['api.user'] || config?.api?.user || '').trim();
  const password =
    String(config?.['api.password'] || config?.api?.password || '').trim();
  if (!username || !password) {
    throw new Error('Informe api.user e api.password para autenticação do Filhos de Minas.');
  }
  const tokenUrl = `${baseUrl}/auth/token`;
  const body = new URLSearchParams({ username, password }).toString();
  const tokenCurl = buildCurlCommand(
    tokenUrl,
    { 'Content-Type': 'application/x-www-form-urlencoded' },
    body
  );
  const resp = await fetch(tokenUrl, {
    method: 'POST',
    headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
    body
  });
  const text = await resp.text().catch(() => '');
  if (!resp.ok) {
    const error = new Error(
      `HTTP ${resp.status} ${resp.statusText} ao buscar token - ${text || 'sem corpo'}`
    );
    error.curl = tokenCurl;
    throw error;
  }
  let payloadJson = {};
  if (text) {
    try {
      payloadJson = JSON.parse(text);
    } catch (err) {
      const error = new Error(`Resposta do token inválida: ${err.message}`);
      error.curl = tokenCurl;
      throw error;
    }
  }
  const token = payloadJson.access_token || payloadJson.token;
  if (!token) {
    const error = new Error(
      `Token ausente na resposta do Filhos de Minas: ${text || 'vazia'}`
    );
    error.curl = tokenCurl;
    throw error;
  }
  return { token, curl: tokenCurl };
}

async function sendPendingItems(config = {}, options = {}) {
  const limit = Number.isFinite(Number(options.limit)) && options.limit > 0 ? Number(options.limit) : undefined;
  const pending = getPendingItems(MODULE_ID, limit);
  if (!pending || pending.length === 0) {
    console.log('[Filhos de Minas] Nenhum registro pendente para preparar.');
    return { total: 0, payloads: [] };
  }

  const attemptId = buildAttemptId();
  const payloadGroups = buildGroupedPayloads(pending, attemptId);
  const summary = { total: pending.length, sent: 0, errors: 0 };

  safeInitialLogs(attemptId, summary.total);

  const rawUrl = config.baseUrl || process.env.FILHOS_DE_MINAS_API_URL || FALLBACK_API_URL;
  const baseUrl = String(rawUrl || FALLBACK_API_URL).replace(/\/$/, '');
  const endpoint = `${baseUrl}/extrator/importacao`;
  let headers = Object.assign({ 'Content-Type': 'application/json' }, config.headers || {});
  const fatalStatuses = new Set([401, 404]);

  let firstErrorMessage = null;
  let firstErrorDetails = null;
  let fatalError = null;
  let hadToken = false;

  try {
    const tokenInfo = await getFilhosToken(config, baseUrl);
    headers = { ...headers, Authorization: `Bearer ${tokenInfo.token}` };
    safeLogModuleEvent({
      moduleId: MODULE_ID,
      attemptId,
      action: 'authentication',
      status: 'success',
      message: 'Token obtido com sucesso'
    });
    hadToken = true;
    for (const group of payloadGroups) {
      const payload = group.payload;
      const errors = validatePayload(payload);
      if (errors.length > 0) {
        const message = `Campos ausentes: ${errors.join(', ')}`;
        const gestante = payload.gestante || {};
        safeLogModuleEvent({
          moduleId: MODULE_ID,
          attemptId,
          itemId: payload.item_id || null,
          action: 'send.validation',
          status: 'error',
          message,
          details: `Gestante: ${gestante.nome || '—'} (${payload.item_id || '—'})`
        });
        console.error('[Filhos de Minas] Validação falhou', message, payload.item_id || 'unknown');
        summary.errors += group.rowIds.length;
        if (!firstErrorMessage) {
          firstErrorMessage = message;
          firstErrorDetails = payload;
        }
        if (group.rowIds.length) {
          for (const id of group.rowIds) {
            setSyncResponse(MODULE_ID, id, {
              status: 'error',
              message
            });
          }
        }
        if (options.delayMs) await delay(options.delayMs);
        continue;
      }

      let curlText;
      try {
        curlText = buildCurlCommand(endpoint, headers, payload);
        const { resp, text } = await sendRequest(endpoint, headers, payload);
        if (!resp.ok) {
          const message = `HTTP ${resp.status} ${resp.statusText} - ${text || 'sem corpo de resposta'}`;
          safeLogModuleEvent({
            moduleId: MODULE_ID,
            attemptId,
            itemId: payload.item_id || null,
            action: 'send.api_error',
            status: 'error',
            message,
            details: `${curlText}\nResposta: ${text || 'sem corpo de resposta'}`
          });
          console.error('[Filhos de Minas] Envio falhou', resp.status, text || payload);
          summary.errors += group.rowIds.length;
          if (!firstErrorMessage) {
            firstErrorMessage = message;
            firstErrorDetails = text || payload;
          }
          if (group.rowIds.length) {
            for (const id of group.rowIds) {
              setSyncResponse(MODULE_ID, id, {
                status: 'error',
                message
              });
            }
          }
          if (fatalStatuses.has(resp.status)) {
            const fatalMsg = `Parando batch por falha fatal: ${message}`;
            fatalError = new Error(fatalMsg);
            safeLogModuleEvent({
              moduleId: MODULE_ID,
              attemptId,
              action: 'batch.error',
              status: 'error',
              message: fatalMsg,
              details: curlText
            });
            break;
          }
        } else {
          summary.sent += group.rowIds.length;
          console.log('[Filhos de Minas] Enviado com sucesso', payload.item_id || payload.integracao_id || 'unknown');
          let showPayloadSucess = process.env.SHOW_PAYLOAD_SUCCESS_ON_LOGS === 'true';
          let payloadMessage = 'Enviado com sucesso';
          if (showPayloadSucess) {
            payloadMessage += ` - Payload: ${JSON.stringify(payload)}`;
          }
          safeLogModuleEvent({
            moduleId: MODULE_ID,
            attemptId,
            itemId: payload.item_id || null,
            action: 'send.item',
            status: 'success',
            message: payloadMessage
          });
          if (group.rowIds.length) {
            for (const id of group.rowIds) {
              setSyncResponse(MODULE_ID, id, {
                status: 'success',
                message: 'Preparado para envio'
              });
              markQueued(MODULE_ID, id);
            }
          }
        }
      } catch (err) {
        summary.errors += group.rowIds.length;
        console.error('[Filhos de Minas] Erro ao enviar', err);
        if (!firstErrorMessage) {
          firstErrorMessage = err instanceof Error ? err.message : String(err);
          firstErrorDetails = err?.stack || err;
        }
        safeLogModuleEvent({
          moduleId: MODULE_ID,
          attemptId,
          itemId: payload.item_id || null,
          action: 'send.api_error',
          status: 'error',
          message: 'Erro ao enviar (exception)',
          details: `${curlText || 'curl indisponível'}\nErro: ${err?.stack || err}`
        });
        if (group.rowIds.length) {
          for (const id of group.rowIds) {
            setSyncResponse(MODULE_ID, id, {
              status: 'error',
              message: 'Erro ao enviar (exception)'
            });
          }
        }
      }

      if (options.delayMs) await delay(options.delayMs);
    }
  } catch (err) {
    fatalError = err;
    const errorDetails = describeErrorDetails(err);
    if (!firstErrorMessage) {
      const msg = err instanceof Error ? err.message : String(err);
      firstErrorMessage = msg;
      firstErrorDetails = errorDetails;
    }
    if (!hadToken) {
      summary.errors = summary.total;
    } else {
      summary.errors += 1;
    }
    console.error('[Filhos de Minas] Falha fatal durante o processamento', err);
    safeLogModuleEvent({
      moduleId: MODULE_ID,
      attemptId,
      action: 'batch.error',
      status: 'error',
      message: err instanceof Error ? err.message : String(err),
      details: errorDetails
    });
  } finally {
    const summaryDetails = firstErrorMessage
      ? `Primeiro erro: ${firstErrorMessage}${
          firstErrorDetails ? `\n${firstErrorDetails}` : ''
        }`
      : '';

    safeLogModuleEvent({
      moduleId: MODULE_ID,
      attemptId,
      action: 'send.summary',
      status: summary.errors > 0 ? 'error' : 'success',
      message: `Batch finalizado: ${summary.sent}/${summary.total} com ${summary.errors} falhas`,
      details: summaryDetails
    });
  }

  return {
    total: summary.total,
    sent: summary.sent,
    errors: summary.errors,
    payloads: payloadGroups,
    attemptId,
    fatalError
  };
}

module.exports = {
  sendPendingItems
};
