const {
  getLastItemId,
  getMissingNameIds,
  getMissingDtIds,
  persistModuleItems,
  getValidDumEntries
} = require('../../app/core/storage');

const MODULE_ID = 'filhos_de_minas';
const TERMS_ACCEPT_GUIDANCE =
  'Leia e aceite os Termos de Uso (modules/filhos_de_minas/TERMS.md) via GUI ou rode "extrator-esus config-set --module filhos_de_minas --set termsAccepted=true" antes de executar.';
const query = `
WITH ciaps_gestante AS (
    SELECT co_seq_dim_ciap
    FROM tb_dim_ciap
    WHERE nu_ciap IN ('W78','W79','W81','W84','W85')
),
cids_gestante AS (
    SELECT co_seq_dim_cid
    FROM tb_dim_cid
    WHERE nu_cid IN (
        'O11',
        'O120','O121','O122',
        'O13',
        'O140','O141','O149',
        'O150','O151','O159',
        'O16',
        'O200','O208','O209',
        'O210','O211','O212','O218','O219',
        'O220','O221','O222','O223','O224','O225','O228','O229',
        'O230','O231','O232','O233','O234','O235','O239',
        'O299',
        'O300','O301','O302','O308','O309',
        'O311','O312','O318',
        'O320','O321','O322','O323','O324','O325','O326','O328','O329',
        'O330','O331','O332','O333','O334','O335','O336','O337','O338',
        'O752','O753',
        'O990','O991','O992','O993','O994',
        'O240','O241','O242','O243','O244','O249',
        'O25',
        'O260','O261','O263','O264','O265','O268','O269',
        'O280','O281','O282','O283','O284','O285','O288','O289',
        'O290','O291','O292','O293','O294','O295','O296','O298',
        'O339',
        'O340','O341','O342','O343','O344','O345','O346','O347','O348','O349',
        'O350','O351','O352','O353','O354','O355','O356','O357','O358','O359',
        'O360','O361','O362','O363','O365','O366','O367','O368','O369',
        'O40',
        'O410','O411','O418','O419',
        'O430','O431','O438','O439',
        'O440','O441',
        'O460','O468','O469',
        'O470','O471','O479',
        'O48',
        'O995','O996','O997',
        'Z640',
        'O10','O12','O14','O15','O20','O21','O22','O23','O24',
        'O26','O28','O29','O30','O31','O32','O33','O34','O35','O36',
        'O41','O43','O44','O46','O47','O98',
        'Z34','Z35','Z36','Z33',
        'Z340','Z348','Z349',
        'Z350','Z351','Z352','Z353','Z354','Z357','Z358','Z359'
    )
),
cid_multifetal AS (
    SELECT d.co_seq_dim_cid, t.qtd_filhos
    FROM (VALUES
        ('O300', 2),
        ('O301', 3),
        ('O302', 4)
    ) AS t(nu_cid, qtd_filhos)
    JOIN tb_dim_cid d ON d.nu_cid = t.nu_cid
),
cbo_elegivel AS (
    SELECT co_seq_dim_cbo
    FROM tb_dim_cbo
    WHERE nu_cbo LIKE '2251%' OR nu_cbo LIKE '2235%'
),
atendimentos_gestantes AS (
    SELECT DISTINCT
        p.co_seq_fat_atend_ind_problemas,
        a.co_seq_fat_atd_ind,
        a.co_fat_cidadao_pec,
        a.co_dim_tempo_dum,
        a.dt_inicial_atendimento,
        CASE
            WHEN cb1.co_seq_dim_cbo IS NOT NULL THEN a.co_dim_cbo_1
            WHEN cb2.co_seq_dim_cbo IS NOT NULL THEN a.co_dim_cbo_2
        END AS co_dim_cbo,
        p.co_dim_cid,
        p.co_dim_ciap,
        COALESCE(dp1.no_profissional, dp2.no_profissional) AS profissional_nome
    FROM tb_fat_atd_ind_problemas p
    JOIN tb_fat_atendimento_individual a ON a.co_seq_fat_atd_ind = p.co_fat_atd_ind
    LEFT JOIN tb_dim_profissional dp1 ON dp1.co_seq_dim_profissional = a.co_dim_profissional_1
    LEFT JOIN tb_dim_profissional dp2 ON dp2.co_seq_dim_profissional = a.co_dim_profissional_2
    LEFT JOIN cbo_elegivel cb1 ON cb1.co_seq_dim_cbo = a.co_dim_cbo_1
    LEFT JOIN cbo_elegivel cb2 ON cb2.co_seq_dim_cbo = a.co_dim_cbo_2
    LEFT JOIN cids_gestante  cg ON cg.co_seq_dim_cid  = p.co_dim_cid
    LEFT JOIN ciaps_gestante ci ON ci.co_seq_dim_ciap = p.co_dim_ciap
    WHERE
        a.co_dim_tempo_dum IS NOT NULL
        AND (cg.co_seq_dim_cid IS NOT NULL OR ci.co_seq_dim_ciap IS NOT NULL)
        AND (cb1.co_seq_dim_cbo IS NOT NULL OR cb2.co_seq_dim_cbo IS NOT NULL)
),
qtd_filhos_por_gestacao AS (
    SELECT
        ag.co_fat_cidadao_pec,
        ag.co_dim_tempo_dum,
        COALESCE(MAX(mf.qtd_filhos), 1) AS qtd_filhos
    FROM atendimentos_gestantes ag
    LEFT JOIN tb_fat_atd_ind_problemas p2 ON p2.co_fat_atd_ind = ag.co_seq_fat_atd_ind
    LEFT JOIN cid_multifetal mf ON mf.co_seq_dim_cid = p2.co_dim_cid
    GROUP BY ag.co_fat_cidadao_pec, ag.co_dim_tempo_dum
)
SELECT DISTINCT
    ag.co_seq_fat_atd_ind                   AS co_seq_fat_atd_ind,
    c.no_cidadao                            AS nome,
    COALESCE(a.nu_cpf_cidadao, c.nu_cpf, fcp_cid.nu_cpf_cidadao) AS cpf,
    a.nu_cns                                AS cns,
    a.co_fat_cidadao_pec                    AS co_fat_cidadao,
    c.no_mae                               AS mother_name,
    c.dt_nascimento                        AS birth_date,
    t.dt_registro                           AS dum,
    cbo.nu_cbo                              AS cbo,
    dc.nu_cid                               AS cid,
    di.nu_ciap                              AS ciap,
    ag.profissional_nome                               AS profissional_nome,
    ag.dt_inicial_atendimento                 AS dt_inicial_atendimento,
    q.qtd_filhos
FROM atendimentos_gestantes ag
JOIN tb_fat_atendimento_individual a ON a.co_seq_fat_atd_ind = ag.co_seq_fat_atd_ind
LEFT JOIN tb_fat_cidadao_pec fcp ON a.co_fat_cidadao_pec = fcp.co_seq_fat_cidadao_pec
JOIN tb_cidadao c ON fcp.co_cidadao = c.co_seq_cidadao
LEFT JOIN (
    SELECT co_cidadao, MAX(nu_cpf_cidadao) AS nu_cpf_cidadao
    FROM tb_fat_cidadao_pec
    GROUP BY co_cidadao
) fcp_cid ON fcp_cid.co_cidadao = c.co_seq_cidadao
LEFT JOIN tb_dim_cbo cbo ON cbo.co_seq_dim_cbo = ag.co_dim_cbo
LEFT JOIN tb_dim_cid dc ON dc.co_seq_dim_cid = ag.co_dim_cid
LEFT JOIN tb_dim_ciap di ON di.co_seq_dim_ciap = ag.co_dim_ciap
LEFT JOIN tb_dim_tempo t ON t.co_seq_dim_tempo = ag.co_dim_tempo_dum
JOIN qtd_filhos_por_gestacao q
  ON q.co_fat_cidadao_pec = ag.co_fat_cidadao_pec
 AND q.co_dim_tempo_dum   = ag.co_dim_tempo_dum
WHERE t.dt_registro >= CURRENT_DATE - INTERVAL '365' DAY
  AND t.dt_registro <= CURRENT_DATE
ORDER BY c.no_cidadao, t.dt_registro;
`;

async function run(params = {}, config = {}) {
  if (!config?.termsAccepted) {
    throw new Error(`Termos de uso do Filhos de Minas não aceitos. ${TERMS_ACCEPT_GUIDANCE}`);
  }
  const pageSize = Math.min(Math.max(Number(params.pageSize) || 10, 1), 25);
  const page = Math.max(Number(params.page) || 1, 1);
  const offset = (page - 1) * pageSize;
  const baseQuery = query.trim().replace(/;$/, '');
  const filter = (params.filter || '').trim();
  const lastPersistedId = getLastItemId(MODULE_ID);
  const missingNameIds = getMissingNameIds(MODULE_ID);
  const missingDtIds = getMissingDtIds(MODULE_ID);
  const repairIds = [...new Set([...missingNameIds, ...missingDtIds])];
  const minId = Number.isFinite(Number(lastPersistedId)) ? Number(lastPersistedId) : null;

  const dbType = (config['dw.type'] || '').toLowerCase();
  if (!['postgres', 'oracle'].includes(dbType)) {
    throw new Error('Configuração dw.type inválida. Use postgres ou oracle.');
  }

  const required = ['dw.host', 'dw.port', 'dw.user', 'dw.password'];
  const missing = required.filter((k) => !config[k]);
  if (missing.length) {
    throw new Error(`Configurações ausentes: ${missing.join(', ')}`);
  }

  const connInfo = {
    host: config['dw.host'],
    port: Number(config['dw.port']),
    user: config['dw.user'],
    password: config['dw.password'],
    database: null,
    service: null
  };

  if (dbType === 'postgres') {
    if (!config['dw.database']) {
      throw new Error('Informe dw.database (nome da base Postgres).');
    }
    connInfo.database = config['dw.database'];
    const result = await runPostgres({
      baseQuery,
      connInfo,
      offset,
      pageSize,
      page,
      filter,
      minId,
      repairIds
    });
    const persisted = persistModuleItems(MODULE_ID, result.allNewRows || result.rows);
    if (result.repairedRows && result.repairedRows.length) {
      persistModuleItems(MODULE_ID, result.repairedRows);
    }
    return { ...result, incrementalFromId: minId, persisted };
  }
  if (!config['dw.service']) {
    throw new Error('Informe dw.service (service name/SID Oracle).');
  }
  connInfo.database = config['dw.service'];
  const result = await runOracle({
    baseQuery,
    connInfo,
    offset,
    pageSize,
    page,
    filter,
    minId,
    repairIds
  });
  const persisted = persistModuleItems(MODULE_ID, result.allNewRows || result.rows);
  if (result.repairedRows && result.repairedRows.length) {
    persistModuleItems(MODULE_ID, result.repairedRows);
  }
  return { ...result, incrementalFromId: minId, persisted };
}

function buildPgWhere(filter, minId) {
  const conditions = [];
  const params = [];
  if (minId !== null && minId !== undefined) {
    conditions.push(`co_seq_fat_atd_ind > $${params.length + 1}`);
    params.push(minId);
  }
  if (filter) {
    const pattern = `%${filter}%`;
    const start = params.length + 1;
    conditions.push(
      `(nome ILIKE $${start} OR cid ILIKE $${start + 2} OR ciap ILIKE $${start + 3})`
    );
    params.push(pattern, pattern, pattern, pattern);
  }
  const clause = conditions.length ? `WHERE ${conditions.join(' AND ')}` : '';
  return { clause, params };
}

async function runPostgres({
  baseQuery,
  connInfo,
  offset,
  pageSize,
  page,
  filter,
  minId,
  repairIds = []
}) {
  let client;
  try {
    const { Client } = require('pg');
    client = new Client({
      host: connInfo.host,
      port: connInfo.port,
      user: connInfo.user,
      password: connInfo.password,
      database: connInfo.database
    });
    await client.connect();
    const { clause, params: whereParams } = buildPgWhere(filter, minId);
    const paginatedQuery = `SELECT * FROM (${baseQuery}) sub ${clause} ORDER BY co_seq_fat_atd_ind ASC LIMIT $${
      whereParams.length + 1
    } OFFSET $${whereParams.length + 2}`;
    const params = [...whereParams, pageSize, offset];

    const res = await client.query(paginatedQuery, params);
    const countSql = `SELECT COUNT(*) AS total FROM (${baseQuery}) AS sub ${clause}`;
    const countRes = await client.query(countSql, whereParams);
    const total = Number(countRes.rows[0].total || 0);
    const rows = res.rows.map(normalizeRow);
    const summary = await computeSummaryPostgres(client, baseQuery, clause, whereParams);
    const allNewRows = await fetchAllNewRowsPostgres(client, baseQuery, minId);
    const storedDumCandidates = getValidDumEntries(MODULE_ID, 2000);
    const dumFallback = buildValidDumMap([...allNewRows, ...storedDumCandidates]);
    applyDumFallback(allNewRows, dumFallback);
    applyDumFallback(rows, dumFallback);
    const repairedRows = repairIds.length
      ? await fetchByIdsPostgres(client, baseQuery, repairIds)
      : [];
    applyDumFallback(repairedRows, dumFallback);
    return { query: baseQuery, page, pageSize, total, rows, summary, allNewRows, repairedRows };
  } catch (err) {
    if (err.code === 'MODULE_NOT_FOUND') {
      throw new Error('Dependência pg não instalada. Rode npm install pg.');
    }
    throw err;
  } finally {
    if (client) {
      await client.end().catch(() => {});
    }
  }
}

function buildOracleWhere(filter, minId) {
  const params = {};
  const conditions = [];
  if (minId !== null && minId !== undefined) {
    params.minId = minId;
    conditions.push('co_seq_fat_atd_ind > :minId');
  }
  if (filter) {
    const pattern = `%${filter}%`;
    params.f1 = pattern;
    params.f2 = pattern;
    params.f3 = pattern;
    params.f4 = pattern;
    conditions.push(
      'LOWER(nome) LIKE LOWER(:f1) OR LOWER(cid) LIKE LOWER(:f3) OR LOWER(ciap) LIKE LOWER(:f4)'
    );
  }
  const clause = conditions.length ? `WHERE ${conditions.join(' AND ')}` : '';
  return { clause, params };
}

async function runOracle({
  baseQuery,
  connInfo,
  offset,
  pageSize,
  page,
  filter,
  minId,
  repairIds = []
}) {
  let connection;
  try {
    const oracledb = require('oracledb');
    const { clause, params: whereParams } = buildOracleWhere(filter, minId);
    const paginatedQuery = `SELECT * FROM (${baseQuery}) sub ${clause}\nORDER BY co_seq_fat_atd_ind ASC\nOFFSET :offset ROWS FETCH NEXT :limit ROWS ONLY`;
    connection = await oracledb.getConnection({
      user: connInfo.user,
      password: connInfo.password,
      connectString: `${connInfo.host}:${connInfo.port}/${connInfo.database}`
    });
    const res = await connection.execute(
      paginatedQuery,
      { offset, limit: pageSize, ...whereParams },
      { outFormat: oracledb.OUT_FORMAT_OBJECT }
    );
    const countSql = `SELECT COUNT(*) AS TOTAL FROM (${baseQuery}) sub ${clause}`;
    const countRes = await connection.execute(countSql, whereParams, {
      outFormat: oracledb.OUT_FORMAT_OBJECT
    });
    const total = Number(countRes.rows?.[0]?.TOTAL || 0);
    const rows = (res.rows || []).map(normalizeRow);
    const summary = await computeSummaryOracle(connection, baseQuery, clause, whereParams);
    const allNewRows = await fetchAllNewRowsOracle(connection, baseQuery, minId);
    const storedDumCandidates = getValidDumEntries(MODULE_ID, 2000);
    const dumFallback = buildValidDumMap([...allNewRows, ...storedDumCandidates]);
    applyDumFallback(allNewRows, dumFallback);
    applyDumFallback(rows, dumFallback);
    const repairedRows = repairIds.length
      ? await fetchByIdsOracle(connection, baseQuery, repairIds)
      : [];
    applyDumFallback(repairedRows, dumFallback);
    return { query: baseQuery, page, pageSize, total, rows, summary, allNewRows, repairedRows };
  } catch (err) {
    if (err.code === 'MODULE_NOT_FOUND') {
      throw new Error('Dependência oracledb não instalada. Rode npm install oracledb.');
    }
    throw err;
  } finally {
    if (connection) {
      try {
        await connection.close();
      } catch (e) {
        // ignore
      }
    }
  }
}

function normalizeRow(row) {
  if (!row) return {};
  const get = (k) => row[k] ?? row[k.toUpperCase()] ?? row[k.toLowerCase()];
  const cpf = get('cpf');
  const cns = get('cns');
  const docTipo = cpf ? 'cpf' : cns ? 'cns' : null;
  return {
    co_seq_fat_atd_ind: get('co_seq_fat_atd_ind'),
    nome: get('nome'),
    doc_tipo: docTipo,
    cpf,
    cns,
    co_fat_cidadao: get('co_fat_cidadao'),
    mother_name: get('mother_name'),
    birth_date: get('birth_date'),
    professional_name: get('profissional_nome'),
    dt_inicial_atendimento: get('dt_inicial_atendimento'),
    dum: get('dum'),
    cbo: get('cbo'),
    cid: get('cid'),
    ciap: get('ciap'),
    qtd_filhos: get('qtd_filhos')
  };
}

async function computeSummaryPostgres(client, baseQuery, clause, params) {
  const sumSql = `
    SELECT
      COUNT(*) AS total,
      COUNT(DISTINCT nome) AS gestantes,
      COUNT(DISTINCT cid || '|' || ciap) AS combos,
      AVG(qtd_filhos)::numeric AS media
    FROM (${baseQuery}) sub
    ${clause || ''}
  `;
  const res = await client.query(sumSql, clause ? params : []);
  const row = res.rows?.[0] || {};
  return {
    total: Number(row.total || 0),
    gestantes: Number(row.gestantes || 0),
    uniqueCids: Number(row.combos || 0),
    mediaFilhos: row.media ? Number(Number(row.media).toFixed(2)) : 0
  };
}

async function computeSummaryOracle(connection, baseQuery, clause, params) {
  const sumSql = `
    SELECT
      COUNT(*) AS TOTAL,
      COUNT(DISTINCT nome) AS GESTANTES,
      COUNT(DISTINCT cid || '|' || ciap) AS COMBOS,
      AVG(qtd_filhos) AS MEDIA
    FROM (${baseQuery}) sub
    ${clause || ''}
  `;
  const res = await connection.execute(sumSql, params || {}, {
    outFormat: oracledb.OUT_FORMAT_OBJECT
  });
  const row = res.rows?.[0] || {};
  return {
    total: Number(row.TOTAL || 0),
    gestantes: Number(row.GESTANTES || 0),
    uniqueCids: Number(row.COMBOS || 0),
    mediaFilhos: row.MEDIA ? Number(Number(row.MEDIA).toFixed(2)) : 0
  };
}

async function fetchAllNewRowsPostgres(client, baseQuery, minId) {
  const sql =
    minId !== null && minId !== undefined
      ? `SELECT * FROM (${baseQuery}) sub WHERE co_seq_fat_atd_ind > $1 ORDER BY co_seq_fat_atd_ind ASC`
      : `SELECT * FROM (${baseQuery}) sub ORDER BY co_seq_fat_atd_ind ASC`;
  const res = await client.query(sql, minId !== null && minId !== undefined ? [minId] : []);
  return res.rows.map(normalizeRow);
}

async function fetchAllNewRowsOracle(connection, baseQuery, minId) {
  const clause =
    minId !== null && minId !== undefined ? 'WHERE co_seq_fat_atd_ind > :minId' : '';
  const sql = `SELECT * FROM (${baseQuery}) sub ${clause} ORDER BY co_seq_fat_atd_ind ASC`;
  const res = await connection.execute(
    sql,
    minId !== null && minId !== undefined ? { minId } : {},
    { outFormat: connection.constructor.OUT_FORMAT_OBJECT || 4002 }
  );
  return (res.rows || []).map(normalizeRow);
}

async function fetchByIdsPostgres(client, baseQuery, ids) {
  if (!ids || ids.length === 0) return [];
  const placeholders = ids.map((_, i) => `$${i + 1}`).join(',');
  const sql = `SELECT * FROM (${baseQuery}) sub WHERE co_seq_fat_atd_ind IN (${placeholders})`;
  const res = await client.query(sql, ids);
  return res.rows.map(normalizeRow);
}

async function fetchByIdsOracle(connection, baseQuery, ids) {
  if (!ids || ids.length === 0) return [];
  const binds = ids.reduce((acc, id, idx) => ({ ...acc, [`id${idx}`]: id }), {});
  const placeholders = ids.map((_, i) => `:id${i}`).join(',');
  const sql = `SELECT * FROM (${baseQuery}) sub WHERE co_seq_fat_atd_ind IN (${placeholders})`;
  const res = await connection.execute(sql, binds, {
    outFormat: connection.constructor.OUT_FORMAT_OBJECT || 4002
  });
  return (res.rows || []).map(normalizeRow);
}

const FUTURE_DUM_YEAR = 3000;

function parseDumValue(value) {
  if (!value) return null;
  if (value instanceof Date) return value;
  const parsed = new Date(value);
  return Number.isNaN(parsed.getTime()) ? null : parsed;
}

function isFutureDumValue(value) {
  const parsed = parseDumValue(value);
  if (!parsed) return false;
  const year = parsed.getUTCFullYear();
  return Number.isFinite(year) && year >= FUTURE_DUM_YEAR;
}

function getDumLookupKey(row) {
  if (!row) return null;
  if (row.co_fat_cidadao !== undefined && row.co_fat_cidadao !== null) {
    return `cid:${row.co_fat_cidadao}`;
  }
  return null;
}

function buildValidDumMap(rows = []) {
  const map = new Map();
  if (!rows || rows.length === 0) return map;
  for (const row of rows) {
    const key = getDumLookupKey(row);
    if (!key) continue;
    const value = row?.dum;
    if (!value || isFutureDumValue(value)) continue;
    if (!map.has(key)) {
      map.set(key, value);
    }
  }
  return map;
}

function applyDumFallback(rows = [], fallback) {
  if (!rows || rows.length === 0 || !fallback || fallback.size === 0) return;
  for (const row of rows) {
    if (!row || !isFutureDumValue(row.dum)) continue;
    const key = getDumLookupKey(row);
    if (!key) continue;
    const fallbackDum = fallback.get(key);
    if (fallbackDum) {
      row.dum = fallbackDum;
    }
  }
}

async function testConnection() {
  console.log('[stub] Teste de conexão Filhos de Minas');
  return { ok: true };
}

async function validateConfig(config) {
  const required = [
    'dw.host',
    'dw.user',
    'dw.password',
    'api.user',
    'api.password',
    'user.name',
    'user.email',
    'user.phone',
    'user.city',
    'user.role'
  ];
  const missing = required.filter((key) => !config[key]);
  return { ok: missing.length === 0, missing };
}

module.exports = {
  run,
  testConnection,
  validateConfig,
  query
};
