# Como construir um módulo para o Extrator PEC e-SUS

## Estrutura básica
```
modules/
  <id-do-modulo>/
    module.json      # manifesto
    index.js         # código do módulo
    README.md        # documentação específica (opcional, recomendado)
```

### module.json (manifesto)
Campos obrigatórios:
- `id`: identificador único (sem espaços).
- `name`: nome amigável.
- `version`: versão semântica do módulo.
- `minAppVersion`: versão mínima do core necessária.
- `requiredSettings`: array de configurações que o usuário precisa informar (chave, tipo, título, descrição e se é opcional).
- `commands`: comandos suportados (`gui` e/ou `cli`, ex.: `["run"]`).

Exemplo:
```json
{
  "id": "meu_modulo",
  "name": "Meu Módulo",
  "version": "1.0.0",
  "minAppVersion": "1.0.0",
  "requiredSettings": [
    { "key": "dw.host", "type": "string", "title": "Host", "description": "Host do DW" },
    { "key": "dw.user", "type": "string", "title": "Usuário" },
    { "key": "dw.password", "type": "secret", "title": "Senha" }
  ],
  "commands": {
    "gui": ["run"],
    "cli": ["run"]
  }
}
```

### index.js (API do módulo)
Exporta obrigatoriamente:
- `async function run(params, config)`: executa a extração (GUI/CLI). Deve validar `config` conforme `requiredSettings` e retornar objetos/linhas para a UI/CLI.
- `async function testConnection()`: teste rápido de conexão (opcional, mas recomendado).
- `async function validateConfig(config)`: retorna `{ ok: boolean, missing: string[] }` para informar campos faltantes.
- `query` (opcional): string da consulta principal, útil para debug.

Exemplo mínimo:
```js
async function run(params = {}, config = {}) {
  if (!config['dw.host']) throw new Error('Falta dw.host');
  // ... conectar ao DW, executar query, transformar linhas
  return { rows: [], total: 0, page: 1, pageSize: 10 };
}

async function testConnection() { return { ok: true }; }

async function validateConfig(config) {
  const required = ['dw.host', 'dw.user', 'dw.password'];
  const missing = required.filter((k) => !config[k]);
  return { ok: missing.length === 0, missing };
}

module.exports = { run, testConnection, validateConfig };
```

## Boas práticas
- **Validação de config:** implemente `validateConfig` e cheque `requiredSettings` no `run`.
- **Erros claros:** lance `Error` com mensagens úteis; a UI mostra o texto ao usuário.
- **Paginação/filtro:** se retornar muitas linhas, ofereça `page`, `pageSize`, `filter` em `params` e responda com `{ rows, total, page, pageSize }`.
- **Resumo:** inclua um `summary` (totais, médias) para exibir na GUI.
- **Persistência local:** use funções do core (`app/core/storage.js`) se precisar gravar no SQLite, mantendo incremental por ID.
- **Sem dependências externas desnecessárias:** prefira libs já presentes; se precisar de nova, declare em `package.json`.
- **Compatibilidade:** respeite `minAppVersion` e atualize `version` do módulo quando mudar comportamento.

## Testes e CLI
- Use `npm run cli -- run --module=<id> --param1=...` para testar via linha de comando.
- Logs/erros aparecerão no terminal e na GUI.

## Publicação/atualização
- Atualize `version` e publique o pacote do módulo conforme o pipeline de distribuição.
- Se houver fixes de segurança, marque `type: "security"` e `mandatory: true` no catálogo de updates para bloquear execuções antigas.
