const path = require('path');
const fs = require('fs');
require('dotenv').config({
  path: path.join(__dirname, '../../.env'),
  override: true
});
const { app, BrowserWindow, ipcMain, dialog } = require('electron');
const { listModules, getModuleManifest, runModule } = require('../core/modules');
const { checkCoreUpdates, autoUpdateCore } = require('../core/updates');
const { getModuleConfig, updateModuleConfig } = require('../core/config');
const {
  runSql,
  listModuleItems,
  clearModuleItems,
  getModuleLogSummary,
  getModuleLogs,
  getLogMessages
} = require('../core/storage');
const appPkg = require('../../package.json');
const modulesDir = path.join(__dirname, '../../modules');
const { sendPendingItems } = require('../../modules/filhos_de_minas/sender');

let mainWindow;

function createWindow() {
  mainWindow = new BrowserWindow({
    width: 1200,
    height: 800,
    webPreferences: {
      preload: path.join(__dirname, 'preload.js')
    }
  });

  mainWindow.loadFile(path.join(__dirname, '../renderer/index.html'));

  mainWindow.on('closed', () => {
    mainWindow = null;
  });
}

app.on('ready', createWindow);

app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit();
  }
});

app.on('activate', () => {
  if (mainWindow === null) {
    createWindow();
  }
});

ipcMain.handle('modules:list', async () => listModules());

ipcMain.handle('modules:manifest', async (_event, moduleId) =>
  getModuleManifest(moduleId)
);

ipcMain.handle('modules:terms', async (_event, moduleId) => {
  if (!moduleId) throw new Error('moduleId é obrigatório.');
  const termsPath = path.join(modulesDir, moduleId, 'TERMS.md');
  if (!fs.existsSync(termsPath)) return '';
  return fs.readFileSync(termsPath, 'utf-8');
});

ipcMain.handle('modules:filhos-de-minas:send', async () => {
  const moduleId = 'filhos_de_minas';
  const config = getModuleConfig(moduleId);
  if (!config?.termsAccepted) {
    throw new Error(
      'Termos de uso do Filhos de Minas não aceitos. Aceite-os antes de enviar os dados.'
    );
  }
  return sendPendingItems(config, { limit: config['send.limit'] });
});

ipcMain.handle('modules:filhos-de-minas:logs', async (_event, options = {}) => {
  const moduleId = 'filhos_de_minas';
  return {
    summary: getModuleLogSummary(moduleId, options.limitAttempts || 10),
    entries: getModuleLogs({
      moduleId,
      limit: options.limitEntries || 200,
      attemptId: options.attemptId,
      status: options.status || 'all',
      search: options.search || '',
      action: options.action || 'all'
      ,
      messageTerm: options.message || ''
    })
  };
});

ipcMain.handle('modules:filhos-de-minas:log-messages', async (_event, options = {}) => {
  const moduleId = 'filhos_de_minas';
  return getLogMessages({
    moduleId,
    action: options.action || null,
    attemptId: options.attemptId || null
  });
});

ipcMain.handle('modules:run', async (_event, moduleId, params) =>
  runModule(moduleId, params)
);

ipcMain.handle('modules:config:get', async (_event, moduleId) =>
  getModuleConfig(moduleId)
);

ipcMain.handle('modules:config:update', async (_event, moduleId, updates) =>
  updateModuleConfig(moduleId, updates)
);


ipcMain.handle('updates:core', async () => checkCoreUpdates());

ipcMain.handle('updates:core:auto', async (_event, options = {}) => {
  return autoUpdateCore(options);
});

ipcMain.handle('updates:alert', async (_event, message) => {
  await dialog.showMessageBox({
    type: 'info',
    title: 'Atualização',
    message
  });
});

ipcMain.handle('sqlite:run', async (_event, sql) => {
  if (!sql) throw new Error('SQL vazio.');
  return runSql(sql);
});

ipcMain.handle('modules:local-list', async (_event, moduleId, options = {}) => {
  if (!moduleId) throw new Error('moduleId é obrigatório.');
    return listModuleItems({
      moduleId,
      filter: options.filter || '',
      queued: options.queued || 'all',
      status: options.status || 'all',
      docType: options.docType || 'all',
      page: options.page || 1,
      pageSize: options.pageSize || 25,
      orderBy: options.orderBy,
      order: options.order
    });
  });

ipcMain.handle('modules:local-clear', async (_event, moduleId) => {
  if (!moduleId) throw new Error('moduleId é obrigatório.');
  return clearModuleItems(moduleId);
});

ipcMain.handle('core:version', () => app.getVersion?.() || appPkg.version || 'dev');
