const fs = require('fs');
const path = require('path');
const { Readable } = require('stream');
const { pipeline } = require('stream/promises');
const { spawn } = require('child_process');
const os = require('os');
const semver = require('semver');
const jwt = require('jsonwebtoken');
const appPkg = require('../../package.json');
const { loadModulesConfig, GLOBAL_CONFIG_MODULE_ID } = require('./config');
const { resolveUpdatesDirRoot } = require('./paths');
let electronAppPath = null;
try {
  const { app } = require('electron');
  electronAppPath = app?.getPath?.('userData') || null;
} catch (err) {
  // CLI mode: Electron não está disponível
}

const useMockUpdates = process.env.USE_UPDATE_MOCK === 'true';
const sharedKey = process.env.EXTRATOR_SHARED_KEY;
const coreDownloadBase =
  process.env.UPDATE_CORE_DOWNLOAD_BASE ||
  'https://updates.extrator-pec-esus.mg.gov.br/extrator/core';
const downloadWorkdir =
  process.env.UPDATE_DOWNLOAD_DIR ||
  (electronAppPath
    ? path.join(electronAppPath, 'updates', 'core')
    : path.join(resolveUpdatesDirRoot(), 'core'));
function resolveCoreUpdateUrl() {
  const modulesConfig = loadModulesConfig();
  const globalConfig = modulesConfig[GLOBAL_CONFIG_MODULE_ID] || {};
  const overrideUrl = globalConfig['core.updateUrl'];
  const sourceUrl = overrideUrl || process.env.UPDATE_CORE_URL || '';
  const normalized = String(sourceUrl || '').trim();
  return normalized || null;
}

const updateServiceMock = {
  core: {
    id: 'extrator-core',
    name: 'Extrator PEC e-SUS – Core',
    currentVersion: appPkg.version,
    latestVersion: '1.2.0',
    updates: [
      {
        version: '1.0.1',
        type: 'security',
        title: 'Correção de vulnerabilidade na criptografia local',
        description: 'Ajusta a forma de criptografar as credenciais armazenadas no disco.',
        mandatory: true
      },
      {
        version: '1.1.0',
        type: 'business',
        title: 'Suporte a novos módulos estaduais',
        description: 'Adiciona suporte a novos tipos de módulos de extração.',
        mandatory: false
      },
      {
        version: '1.2.0',
        type: 'business',
        title: 'Melhorias de UX no rack de módulos',
        description: 'Melhora a visualização de status, badges e mensagens de erro no rack.',
        mandatory: false
      }
    ],
    artifacts: {
      win32: {
        url: 'https://updates.extrator-pec-esus.mg.gov.br/extrator/core/1.2.0/extrator-pec-esus-Setup-1.2.0.exe',
        type: 'installer'
      },
      linux: {
        url: 'https://updates.extrator-pec-esus.mg.gov.br/extrator/core/1.2.0/extrator-pec-esus-1.2.0.AppImage',
        type: 'appimage'
      },
      darwin: {
        url: 'https://updates.extrator-pec-esus.mg.gov.br/extrator/core/1.2.0/extrator-pec-esus-1.2.0.dmg',
        type: 'installer'
      }
    }
  }
};

function normalizeVersion(value) {
  const normalized = semver.valid(value) ? value : semver.coerce(String(value || '0.0.0'))?.version;
  return normalized || '0.0.0';
}

function normalizeArtifacts(entry = {}) {
  const artifacts = entry.artifacts || entry.files;
  const normalized = {};

  const pushArtifact = (platformKey, artifact) => {
    if (!artifact) return;
    const key = String(platformKey || 'generic').toLowerCase();
    const payload = Array.isArray(artifact)
      ? artifact.map((a) => (typeof a === 'string' ? { url: a } : a))
      : [artifact];
    normalized[key] = [...(normalized[key] || []), ...payload.map((p) => ({ ...p, url: p.url || p.href }))];
  };

  if (artifacts) {
    Object.entries(artifacts).forEach(([platform, artifact]) => pushArtifact(platform, artifact));
  }

  const releases = Array.isArray(entry.releases) ? entry.releases : [];
  releases.forEach((rel) => {
    const relVersion = normalizeVersion(rel.version);
    const apps = rel.applications || rel.files || [];
    apps.forEach((app) => {
      if (!app) return;
      const platformKey = app.platform || app.os || app.target || 'generic';
      pushArtifact(platformKey, { ...app, version: relVersion });
    });
  });

  const keys = Object.keys(normalized);
  if (keys.length === 0) return null;

  const simplified = {};
  keys.forEach((k) => {
    const list = normalized[k].filter(Boolean);
    simplified[k] = list.length === 1 ? list[0] : list;
  });
  return simplified;
}

function resolveArtifactForPlatform(entry, platform = process.platform, arch = process.arch, targetVersion = null) {
  const artifacts = entry?.artifacts;
  if (!artifacts) return null;

  const platformArtifact = artifacts[`${platform}-${arch}`] || artifacts[platform] || artifacts.generic || artifacts.default;
  if (!platformArtifact) return null;

  const pick = (items) => {
    if (!Array.isArray(items)) {
      return items;
    }
    const filtered = targetVersion
      ? items.find((item) => item.version === targetVersion && (!item.arch || item.arch === arch))
      : null;
    if (filtered) return filtered;
    return items.find((item) => !item.arch || item.arch === arch) || items[0];
  };

  const matched = pick(platformArtifact);
  return matched ? { ...matched, url: matched.url || matched.href } : null;
}

function evaluateUpdates(currentVersion, updateEntry) {
  const curr = normalizeVersion(currentVersion);
  const normalizedUpdates = (updateEntry.updates || []).map((u) => ({
    ...u,
    version: normalizeVersion(u.version)
  }));
  const updatesAbove = normalizedUpdates.filter((u) => semver.gt(u.version, curr));
  const securityUpdates = updatesAbove.filter(
    (u) => u.type === 'security' && u.mandatory === true
  );
  const businessUpdates = updatesAbove.filter((u) => u.type === 'business');
  const latest = normalizeVersion(updateEntry.latestVersion);
  const targetVersion =
    updatesAbove.sort((a, b) => semver.rcompare(a.version, b.version))[0]?.version ||
    (semver.gt(latest, curr) ? latest : null);

  if (securityUpdates.length > 0 && semver.valid(curr)) {
    return {
      status: 'security_block',
      updates: { security: securityUpdates, business: [] },
      targetVersion,
      latestVersion: latest
    };
  }

  if (semver.gt(latest, curr)) {
    return {
      status: 'update_available',
      updates: { security: [], business: businessUpdates },
      targetVersion,
      latestVersion: latest
    };
  }

  return { status: 'ok', updates: { security: [], business: [] }, targetVersion: null, latestVersion: latest };
}

function normalizeUpdateEntry(entry = {}) {
  const releases = Array.isArray(entry.releases) ? entry.releases : entry.updates || [];
  return {
    ...entry,
    updates: releases.map((r) => ({
      version: normalizeVersion(r.version),
      type: r.type,
      title: r.title,
      description: r.description,
      mandatory: !!r.mandatory
    })),
    latestVersion: normalizeVersion(entry.latestVersion || entry.latest_version || entry.currentVersion),
    artifacts: normalizeArtifacts(entry)
  };
}

function resolveKey() {
  if (!sharedKey) return null;
  // Se a chave estiver em base64, decodifica para bytes (requisito HS256 da API)
  try {
    const buf = Buffer.from(sharedKey, 'base64');
    if (buf.length > 0) return buf;
  } catch (err) {
    // fallback para texto puro
  }
  return sharedKey;
}

function collectModulesPersonalData() {
  const modulesConfig = loadModulesConfig();
  const entries = [];
  Object.entries(modulesConfig).forEach(([moduleId, settings]) => {
    if (!settings || typeof settings !== 'object') return;
    const data = {};
    Object.entries(settings).forEach(([key, value]) => {
      if (!key.startsWith('user.')) return;
      if (value === undefined || value === null) return;
      if (typeof value === 'string' && value.trim() === '') return;
      data[key] = value;
    });
    if (Object.keys(data).length > 0) {
      entries.push({ moduleId, personalData: data });
    }
  });
  return entries;
}

function buildCoreUpdatePayload() {
  return { modulesPersonalData: collectModulesPersonalData() };
}

function buildAuthHeader(payload) {
  const key = resolveKey();
  if (!key) return {};
  try {
    const token = jwt.sign(payload || {}, key, { expiresIn: '10m' });
    return { Authorization: `Bearer ${token}` };
  } catch (err) {
    return {};
  }
}

async function fetchJson(url, payload = null) {
  const headers = {
    Accept: 'application/json',
    ...buildAuthHeader(payload)
  };
  const res = await fetch(url, { headers });
  if (!res.ok) throw new Error(`HTTP ${res.status}`);
  return res.json();
}

function buildCurl(url, payload = null) {
  const headers = {
    Accept: 'application/json',
    ...buildAuthHeader(payload)
  };
  const parts = [`curl --location '${url}'`];
  Object.entries(headers).forEach(([k, v]) => parts.push(`--header '${k}: ${v}'`));
  return parts.join(' \\\n');
}

function fileNameFromUrl(url) {
  try {
    const parsed = new URL(url);
    const pathname = parsed.pathname || '';
    const clean = pathname.split('/').filter(Boolean);
    return clean[clean.length - 1] || null;
  } catch (err) {
    return null;
  }
}

function resolveCoreDownload(entry, version, platform = process.platform, arch = process.arch) {
  if (!version) return null;
  const artifact = resolveArtifactForPlatform(entry, platform, arch, version);
  if (artifact?.url) {
    return {
      ...artifact,
      url: artifact.url,
      platform,
      arch,
      version,
      fileName: artifact.fileName || fileNameFromUrl(artifact.url)
    };
  }

  const url = getDownloadUrlForCore(version, platform, arch);
  if (!url) return null;
  return {
    url,
    platform,
    arch,
    version,
    type: platform === 'win32' ? 'installer' : 'archive',
    fileName: fileNameFromUrl(url)
  };
}

async function ensureDir(dirPath) {
  await fs.promises.mkdir(dirPath, { recursive: true });
  return dirPath;
}

async function downloadToFile(url, destinationDir, fileNameOverride = null) {
  if (!url) throw new Error('URL de download não informada.');
  const targetDir = await ensureDir(destinationDir || downloadWorkdir);
  const res = await fetch(url);
  if (!res.ok) throw new Error(`Download falhou (HTTP ${res.status})`);
  const fileName = fileNameOverride || fileNameFromUrl(url) || `update-${Date.now()}`;
  const filePath = path.join(targetDir, fileName);
  const bodyStream = res.body?.pipe ? res.body : Readable.fromWeb(res.body);
  await pipeline(bodyStream, fs.createWriteStream(filePath));
  const stat = await fs.promises.stat(filePath);
  return { filePath, size: stat.size };
}

async function launchInstaller(filePath, platform = process.platform, installerInfo = {}) {
  try {
    if (platform !== 'win32') {
      await fs.promises.chmod(filePath, 0o755);
    }
    const args = installerInfo.args || [];
    const child = spawn(filePath, args, { detached: true, stdio: 'ignore', windowsHide: false });
    child.unref();
    return true;
  } catch (err) {
    return false;
  }
}

async function checkCoreUpdates(options = {}) {
  const platform = options.platform || process.platform;
  const arch = options.arch || process.arch;
  if (useMockUpdates) {
    const entry = normalizeUpdateEntry(updateServiceMock.core);
    const evaluation = evaluateUpdates(entry.currentVersion, entry);
    const download = evaluation.targetVersion
      ? resolveCoreDownload(entry, evaluation.targetVersion, platform, arch)
      : null;
    return { ...evaluation, download };
  }

  const targetUrl = resolveCoreUpdateUrl();
  if (!targetUrl) {
    return {
      status: 'unreachable',
      updates: { security: [], business: [] },
      error:
        'URL de atualização do core não configurada (UPDATE_CORE_URL ou core.updateUrl).',
      curl: null,
      targetVersion: null,
      latestVersion: null,
      download: null
    };
  }

  const payload = buildCoreUpdatePayload();
  const curl = buildCurl(targetUrl, payload);
  try {
    const raw = await fetchJson(targetUrl, payload);
    const entry = normalizeUpdateEntry(raw);
    const current = appPkg.version || entry.currentVersion || '0.0.0';
    const evaluation = evaluateUpdates(current, entry);
    const download = evaluation.targetVersion
      ? resolveCoreDownload(entry, evaluation.targetVersion, platform, arch)
      : null;
    return { ...evaluation, download, curl };
  } catch (err) {
    return {
      status: 'unreachable',
      updates: { security: [], business: [] },
      error: err.message,
      curl,
      targetVersion: null,
      latestVersion: null,
      download: null
    };
  }
}

async function downloadCoreUpdate(options = {}) {
  const platform = options.platform || process.platform;
  const arch = options.arch || process.arch;
  const baseDir = options.downloadDir || downloadWorkdir;
  const check = await checkCoreUpdates({ platform, arch });

  if (check.status !== 'update_available' && check.status !== 'security_block') {
    return { ...check, downloaded: false };
  }

  if (!check.download?.url) {
    return {
      ...check,
      downloaded: false,
      error: 'Nenhum artefato de download encontrado para este SO.'
    };
  }

  const version = check.targetVersion || check.latestVersion || 'latest';
  const targetDir = path.join(baseDir, version);
  try {
    const { filePath, size } = await downloadToFile(
      check.download.url,
      targetDir,
      check.download.fileName
    );

    return {
      ...check,
      downloaded: true,
      filePath,
      size,
      version,
      download: { ...check.download, filePath }
    };
  } catch (err) {
    return {
      ...check,
      downloaded: false,
      error: err.message,
      version,
      download: null
    };
  }
}

async function autoUpdateCore(options = {}) {
  const platform = options.platform || process.platform;
  const result = await downloadCoreUpdate({ ...options, platform });

  if (!result.downloaded) {
    return { ...result, applied: false };
  }

  if (options.apply === false) {
    return { ...result, applied: false };
  }

  const applied = await launchInstaller(result.filePath, platform, result.download);
  return { ...result, applied };
}

function getDownloadUrlForCore(version, platform = process.platform, arch = process.arch) {
  if (!version) return null;
  const base = coreDownloadBase.replace(/\/$/, '');
  const segments = [base, platform];
  if (arch) segments.push(arch);
  segments.push(`${version}.zip`);
  return segments.join('/');
}

module.exports = {
  checkCoreUpdates,
  downloadCoreUpdate,
  autoUpdateCore,
  getDownloadUrlForCore
};
