const fs = require("fs");
const path = require("path");
const { resolveDataDirRoot } = require("./paths");

let db;

function resolveDbPath() {
  if (process.env.EXTRATOR_DB_PATH) return process.env.EXTRATOR_DB_PATH;

  try {
    // Quando rodando no Electron (main), use userData (writable em build)
    // eslint-disable-next-line global-require
    const { app } = require("electron");
    const electronApp =
      app || (require("electron").remote && require("electron").remote.app);
    if (electronApp?.getPath) {
      return path.join(electronApp.getPath("userData"), "extrator-local.db");
    }
  } catch (err) {
    // Ignora se não estiver em ambiente Electron (ex.: CLI)
  }

  if (process.env.PORTABLE_EXECUTABLE_DIR) {
    return path.join(process.env.PORTABLE_EXECUTABLE_DIR, "extrator-local.db");
  }

  // Fallback para ambiente de desenvolvimento/CLI
  const dataRoot = resolveDataDirRoot();
  return path.join(dataRoot, "extrator-local.db");
}

const dbPath = resolveDbPath();

function normalizeValue(value) {
  if (value instanceof Date) return value.toISOString();
  return value;
}

function normalizeStatus(status) {
  if (!status) return null;
  const s = String(status).toLowerCase();
  if (s === "success" || s === "error") return s;
  return null;
}

function normalizeMessage(msg) {
  if (msg === undefined) return null;
  if (msg === null) return null;
  if (typeof msg === "string") return msg;
  try {
    return JSON.stringify(msg);
  } catch (err) {
    return String(msg);
  }
}

function detectDocType(doc) {
  if (!doc) return null;
  const digits = String(doc).replace(/\D/g, "");
  if (digits.length === 11) return "cpf";
  if (digits.length === 15) return "cns";
  return null;
}

function getDb() {
  if (db) return db;
  fs.mkdirSync(path.dirname(dbPath), { recursive: true });
  let Database;
  try {
    // lazy require to avoid loading sqlite if never used
    // eslint-disable-next-line global-require
    Database = require("better-sqlite3");
  } catch (err) {
    const baseMsg = "Falha ao carregar o driver nativo better-sqlite3.";
    const hintWin =
      process.platform === "win32"
        ? " Reinstale dependências no Windows (não no WSL) e rode electron-rebuild: npm install && npx electron-rebuild -f -w better-sqlite3."
        : "";
    if (err.code === "MODULE_NOT_FOUND") {
      throw new Error(
        `${baseMsg} Dependência ausente. Rode npm install.${hintWin}`
      );
    }
    const invalidBinary = /not a valid Win32 application|wrong ELF class/i.test(
      err.message || ""
    );
    if (invalidBinary) {
      throw new Error(
        `${baseMsg} Binário incompatível com este SO/arch.${hintWin}`
      );
    }
    throw new Error(`${baseMsg} ${err.message}`);
  }
  db = new Database(dbPath);
  bootstrap(db);
  return db;
}

function bootstrap(database) {
  database.exec(`
    CREATE TABLE IF NOT EXISTS module_items (
      module_id TEXT NOT NULL,
      item_id INTEGER NOT NULL,
      co_fat_cidadao TEXT,
      nome TEXT,
      cpf TEXT,
      cns TEXT,
      cpf_ou_cns TEXT,
      doc_tipo TEXT,
      mother_name TEXT,
      birth_date TEXT,
      professional_name TEXT,
      dt_inicial_atendimento TEXT,
      dum TEXT,
      cbo TEXT,
      cid TEXT,
      ciap TEXT,
      qtd_filhos INTEGER,
      queued INTEGER NOT NULL DEFAULT 0,
      status_response TEXT,
      message_response TEXT,
      created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      updated_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      PRIMARY KEY (module_id, item_id)
    );
  CREATE INDEX IF NOT EXISTS idx_module_items_module ON module_items(module_id);
  `);
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN co_fat_cidadao TEXT`);
  } catch (err) {
    // coluna já existe, ignore
  }
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN doc_tipo TEXT`);
  } catch (err) {}
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN cpf TEXT`);
  } catch (err) {}
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN cns TEXT`);
  } catch (err) {}
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN mother_name TEXT`);
  } catch (err) {}
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN birth_date TEXT`);
  } catch (err) {}
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN professional_name TEXT`);
  } catch (err) {}
  try {
    database.exec(
      `ALTER TABLE module_items ADD COLUMN dt_inicial_atendimento TEXT`
    );
  } catch (err) {}
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN status_response TEXT`);
  } catch (err) {}
  try {
    database.exec(`ALTER TABLE module_items ADD COLUMN message_response TEXT`);
  } catch (err) {}
  database.exec(
    `
    CREATE TABLE IF NOT EXISTS module_logs (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      module_id TEXT NOT NULL,
      attempt_id TEXT NOT NULL,
      item_id INTEGER,
      action TEXT,
      status TEXT,
      message TEXT,
      details TEXT,
      created_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP
    );
    CREATE INDEX IF NOT EXISTS idx_module_logs_module ON module_logs(module_id);
  `
  );
  database.exec(
    `
    CREATE TABLE IF NOT EXISTS module_syncs (
      id INTEGER PRIMARY KEY AUTOINCREMENT,
      module_id TEXT NOT NULL,
      attempt_id TEXT NOT NULL UNIQUE,
      started_at TEXT NOT NULL DEFAULT CURRENT_TIMESTAMP,
      ended_at TEXT,
      total INTEGER DEFAULT 0,
      sent INTEGER DEFAULT 0,
      errors INTEGER DEFAULT 0,
      status TEXT,
      message TEXT
    );
    CREATE INDEX IF NOT EXISTS idx_module_syncs_module ON module_syncs(module_id);
  `
  );
}

function getLastItemId(moduleId) {
  const row = getDb()
    .prepare(
      "SELECT MAX(item_id) AS maxId FROM module_items WHERE module_id = ?"
    )
    .get(moduleId);
  return row?.maxId ?? null;
}

function hasMissingNames(moduleId) {
  const row = getDb()
    .prepare(
      "SELECT COUNT(*) AS missing FROM module_items WHERE module_id = ? AND (nome IS NULL OR nome = '')"
    )
    .get(moduleId);
  return Number(row?.missing || 0) > 0;
}

function getMissingNameIds(moduleId, limit = 500) {
  return getDb()
    .prepare(
      `SELECT item_id FROM module_items WHERE module_id = ? AND (nome IS NULL OR nome = '') ORDER BY item_id ASC LIMIT ?`
    )
    .all(moduleId, limit)
    .map((r) => Number(r.item_id))
    .filter((n) => Number.isFinite(n));
}

function getMissingDtIds(moduleId, limit = 500) {
  return getDb()
    .prepare(
      `SELECT item_id FROM module_items WHERE module_id = ? AND (dt_inicial_atendimento IS NULL OR TRIM(dt_inicial_atendimento) = '') ORDER BY item_id ASC LIMIT ?`
    )
    .all(moduleId, limit)
    .map((r) => Number(r.item_id))
    .filter((n) => Number.isFinite(n));
}

function persistModuleItems(moduleId, items = []) {
  if (!items || items.length === 0) {
    return { inserted: 0, updated: 0 };
  }
  const dbInstance = getDb();
  const insertStmt = dbInstance.prepare(`
    INSERT OR IGNORE INTO module_items (
      module_id, item_id, co_fat_cidadao, nome, cpf, cns, cpf_ou_cns, doc_tipo, mother_name, birth_date, professional_name, dt_inicial_atendimento, dum, cbo, cid, ciap, qtd_filhos, queued, status_response, message_response, updated_at
    ) VALUES (
      @module_id, @item_id, @co_fat_cidadao, @nome, @cpf, @cns, @cpf_ou_cns, @doc_tipo, @mother_name, @birth_date, @professional_name, @dt_inicial_atendimento, @dum, @cbo, @cid, @ciap, @qtd_filhos, 0, @status_response, @message_response, CURRENT_TIMESTAMP
    )
  `);
  const updateStmt = dbInstance.prepare(`
    UPDATE module_items SET
      co_fat_cidadao = @co_fat_cidadao,
      nome = @nome,
      cpf = @cpf,
      cns = @cns,
      cpf_ou_cns = @cpf_ou_cns,
      doc_tipo = @doc_tipo,
      mother_name = @mother_name,
      birth_date = @birth_date,
      professional_name = @professional_name,
      dt_inicial_atendimento = @dt_inicial_atendimento,
      dum = @dum,
      cbo = @cbo,
      cid = @cid,
      ciap = @ciap,
      qtd_filhos = @qtd_filhos,
      status_response = COALESCE(@status_response, status_response),
      message_response = COALESCE(@message_response, message_response),
      updated_at = CURRENT_TIMESTAMP
    WHERE module_id = @module_id AND item_id = @item_id
  `);

  let inserted = 0;
  let updated = 0;
  const tx = dbInstance.transaction((batch) => {
    batch.forEach((item) => {
      const payload = {
        module_id: moduleId,
        item_id: Number(item.co_seq_fat_atd_ind),
        co_fat_cidadao: item.co_fat_cidadao,
        nome: item.nome,
        cpf: item.cpf,
        cns: item.cns,
        cpf_ou_cns: item.cpf_ou_cns,
        doc_tipo:
          item.doc_tipo || item.docType || detectDocType(item.cpf_ou_cns),
        mother_name: item.mother_name,
        birth_date: normalizeValue(item.birth_date),
        professional_name: item.professional_name,
        dt_inicial_atendimento: normalizeValue(item.dt_inicial_atendimento),
        dum: normalizeValue(item.dum),
        cbo: item.cbo,
        cid: item.cid,
        ciap: item.ciap,
        qtd_filhos: item.qtd_filhos,
        status_response: normalizeStatus(item.status_response),
        message_response: normalizeMessage(item.message_response),
      };
      const info = insertStmt.run(payload);
      if (info.changes === 1) {
        inserted += 1;
      } else {
        const updInfo = updateStmt.run(payload);
        if (updInfo.changes === 1) {
          updated += 1;
        }
      }
    });
  });

  tx(items);
  return { inserted, updated };
}

function markQueued(moduleId, itemId) {
  return getDb()
    .prepare(
      "UPDATE module_items SET queued = 1, updated_at = CURRENT_TIMESTAMP WHERE module_id = ? AND item_id = ?"
    )
    .run(moduleId, itemId);
}

function setSyncResponse(moduleId, itemId, { status, message } = {}) {
  const statusNormalized = normalizeStatus(status);
  const messageNormalized = normalizeMessage(message);
  return getDb()
    .prepare(
      "UPDATE module_items SET status_response = @status, message_response = @message, updated_at = CURRENT_TIMESTAMP WHERE module_id = @module_id AND item_id = @item_id"
    )
    .run({
      module_id: moduleId,
      item_id: itemId,
      status: statusNormalized,
      message: messageNormalized,
    });
}

function getSyncStats(moduleId) {
  const row = getDb()
    .prepare(
      "SELECT COUNT(*) AS total, MAX(updated_at) AS last_at FROM module_items WHERE module_id = ?"
    )
    .get(moduleId);
  return {
    total: Number(row?.total || 0),
    lastSyncAt: row?.last_at || null,
  };
}

function runSql(sql, params = {}) {
  const dbInstance = getDb();
  const trimmed = String(sql || "")
    .trim()
    .toLowerCase();
  const isSelect = trimmed.startsWith("select") || trimmed.startsWith("with");
  const stmt = dbInstance.prepare(sql);
  if (isSelect) {
    return stmt.all(params);
  }
  const info = stmt.run(params);
  return { changes: info.changes, lastInsertRowid: info.lastInsertRowid };
}

function listModuleItems({
  moduleId,
  filter = "",
  queued = "all",
  status = "all",
  docType = "all",
  page = 1,
  pageSize = 25,
  orderBy = "item_id",
  order = "asc",
}) {
  const dbInstance = getDb();
  const conditions = ["module_id = @moduleId"];
  const params = { moduleId };
  const ORDERABLE_COLUMNS = {
    item_id: "item_id",
    nome: "nome",
    cpf_ou_cns: "cpf_ou_cns",
    dum: "dum",
    cbo: "cbo",
    cid: "cid",
    ciap: "ciap",
    qtd_filhos: "qtd_filhos",
  };
  const requestedOrderBy = typeof orderBy === "string" ? orderBy : "item_id";
  const normalizedOrderBy = ORDERABLE_COLUMNS[requestedOrderBy] || "item_id";
  const requestedOrder = typeof order === "string" ? order : "asc";
  const normalizedOrder =
    requestedOrder.toLowerCase() === "desc" ? "DESC" : "ASC";
  if (filter) {
    params.filter = `%${filter}%`;
    conditions.push(
      "(nome LIKE @filter OR cpf_ou_cns LIKE @filter OR cid LIKE @filter OR ciap LIKE @filter)"
    );
  }
  if (queued === "queued") {
    conditions.push("queued = 1");
  } else if (queued === "pending") {
    conditions.push("queued = 0");
  }
  if (status === "success") {
    conditions.push("status_response = 'success'");
  } else if (status === "error") {
    conditions.push("status_response = 'error'");
  } else if (status === "none") {
    conditions.push("(status_response IS NULL OR status_response = '')");
  }
  if (docType && docType !== "all") {
    conditions.push("doc_tipo = @docType");
    params.docType = docType;
  }
  const where = conditions.length ? `WHERE ${conditions.join(" AND ")}` : "";
  const offset = Math.max(page - 1, 0) * pageSize;
  const rows = dbInstance
    .prepare(
      `
      SELECT *
      FROM module_items
      ${where}
      ORDER BY ${normalizedOrderBy} ${normalizedOrder}
      LIMIT @limit OFFSET @offset
    `
    )
    .all({ ...params, limit: pageSize, offset });

  const normalizedRows = rows.map((row) => {
    const get = (k) => row[k] ?? row[k.toUpperCase()] ?? row[k.toLowerCase()];
    const doc = get("cpf_ou_cns");
    const docTipo = get("doc_tipo") || detectDocType(doc);
    return {
      co_seq_fat_atd_ind: get("item_id"),
      item_id: get("item_id"),
      co_fat_cidadao: get("co_fat_cidadao"),
      nome: get("nome"),
      cpf_ou_cns: doc,
      doc_tipo: docTipo,
      mother_name: get("mother_name"),
      birth_date: get("birth_date"),
      professional_name: get("professional_name"),
      dt_inicial_atendimento: get("dt_inicial_atendimento"),
      dum: get("dum"),
      cbo: get("cbo"),
      cid: get("cid"),
      ciap: get("ciap"),
      qtd_filhos: get("qtd_filhos"),
      cpf: get("cpf"),
      cns: get("cns"),
      queued: get("queued"),
      status_response: get("status_response"),
      message_response: get("message_response"),
    };
  });

  const countRow = dbInstance
    .prepare(`SELECT COUNT(*) AS total FROM module_items ${where}`)
    .get(params);

  const summaryRow = dbInstance
    .prepare(
      `
      SELECT
        COUNT(*) AS total,
        COUNT(DISTINCT COALESCE(nome, '')) AS gestantes,
        COUNT(DISTINCT cid || '|' || ciap) AS combos,
        AVG(qtd_filhos) AS media
      FROM module_items
      ${where}
    `
    )
    .get(params);

  const summary = {
    total: Number(summaryRow?.total || 0),
    gestantes: Number(summaryRow?.gestantes || 0),
    uniqueCids: Number(summaryRow?.combos || 0),
    mediaFilhos: summaryRow?.media
      ? Number(Number(summaryRow.media).toFixed(2))
      : 0,
  };

  return {
    rows: normalizedRows,
    total: Number(countRow?.total || 0),
    page,
    pageSize,
    summary,
  };
}

function logModuleEvent({
  moduleId,
  attemptId,
  itemId = null,
  action = "event",
  status = "info",
  message = "",
  details = "",
} = {}) {
  if (!moduleId || !attemptId) return null;
  const dbInstance = getDb();
  const stmt = dbInstance.prepare(
    `
    INSERT INTO module_logs (module_id, attempt_id, item_id, action, status, message, details)
    VALUES (@moduleId, @attemptId, @itemId, @action, @status, @message, @details)
  `
  );
  return stmt.run({
    moduleId,
    attemptId,
    itemId,
    action,
    status,
    message: normalizeMessage(message),
    details: normalizeMessage(details),
  });
}

function logSyncSessionStart({
  moduleId,
  attemptId,
  total = 0,
  message = null,
} = {}) {
  if (!moduleId || !attemptId) return null;
  const dbInstance = getDb();
  const stmt = dbInstance.prepare(
    `
    INSERT INTO module_syncs (module_id, attempt_id, total, message)
    VALUES (@moduleId, @attemptId, @total, @message)
  `
  );
  const info = stmt.run({
    moduleId,
    attemptId,
    total,
    message: normalizeMessage(message),
  });
  return info.lastInsertRowid;
}

function updateSyncSession(
  sessionId,
  { status, sent, errors, message = null } = {}
) {
  if (!sessionId) return null;
  const dbInstance = getDb();
  const stmt = dbInstance.prepare(
    `
    UPDATE module_syncs
    SET
      ended_at = CURRENT_TIMESTAMP,
      status = @status,
      sent = @sent,
      errors = @errors,
      message = COALESCE(@message, message)
    WHERE id = @sessionId
  `
  );
  return stmt.run({
    sessionId,
    status,
    sent,
    errors,
    message: normalizeMessage(message),
  });
}

function getModuleLogSummary(moduleId, limitAttempts = 10) {
  if (!moduleId) return [];
  const dbInstance = getDb();
  return dbInstance
    .prepare(
      `
      SELECT
        attempt_id,
        MIN(created_at) AS started_at,
        COUNT(*) AS total,
        SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) AS success_count,
        SUM(CASE WHEN status = 'error' THEN 1 ELSE 0 END) AS error_count
      FROM module_logs
      WHERE module_id = @moduleId
      GROUP BY attempt_id
      ORDER BY started_at DESC
      LIMIT @limit
    `
    )
    .all({ moduleId, limit: limitAttempts });
}

function getLogMessages({ moduleId, action, attemptId }) {
  if (!moduleId || !action) return [];
  const dbInstance = getDb();
  const conditions = [
    "module_id = @moduleId",
    "action = @action",
    "message IS NOT NULL",
    "TRIM(message) <> ''",
  ];
  if (attemptId) {
    conditions.push("attempt_id = @attemptId");
  }
  const where = conditions.length ? `WHERE ${conditions.join(" AND ")}` : "";
  return dbInstance
    .prepare(
      `
      SELECT DISTINCT message
      FROM module_logs
      ${where}
      ORDER BY message ASC
      LIMIT 50
    `
    )
    .all({ moduleId, action, attemptId })
    .map((row) => normalizeMessage(row.message));
}

function getModuleLogs({
  moduleId,
  limit = 200,
  attemptId = null,
  status = null,
  search = "",
  action = null,
  messageTerm = "",
} = {}) {
  if (!moduleId) return [];
  const dbInstance = getDb();
  const conditions = ["module_id = @moduleId"];
  const params = { moduleId, limit };
  if (search) {
    params.search = `%${search}%`;
  }
  if (attemptId) {
    conditions.push("attempt_id = @attemptId");
    params.attemptId = attemptId;
  }
  if (status && status !== "all") {
    conditions.push("status = @status");
    params.status = status;
  }
  if (search) {
    conditions.push("(message LIKE @search OR details LIKE @search)");
  }
  if (action && action !== "all") {
    conditions.push("action = @action");
    params.action = action;
  }
  if (messageTerm) {
    conditions.push("message LIKE @message");
    params.message = `%${messageTerm}%`;
  }
  const where = `WHERE ${conditions.join(" AND ")}`;
  return dbInstance
    .prepare(
      `
      SELECT *
      FROM module_logs
      ${where}
      ORDER BY created_at DESC
      LIMIT @limit
    `
    )
    .all(params);
}

function getPendingItems(moduleId, limit) {
  const dbInstance = getDb();
  const limitClause =
    Number.isFinite(Number(limit)) && limit > 0 ? "LIMIT @limit" : "";
  const query = `
      SELECT item_id, nome, cpf, cns, cpf_ou_cns, doc_tipo, mother_name, birth_date, professional_name, dt_inicial_atendimento,dum, cbo, cid, ciap, qtd_filhos
      FROM module_items
      WHERE module_id = @moduleId AND queued = 0
      ORDER BY item_id ASC
      ${limitClause}
    `;
  const params = { moduleId };
  if (limitClause) {
    params.limit = limit;
  }
  const rows = dbInstance.prepare(query).all(params);
  return rows.map((row) => {
    const get = (k) => row[k] ?? row[k.toUpperCase()] ?? row[k.toLowerCase()];
    return {
      item_id: get("item_id"),
      nome: get("nome"),
      cpf: get("cpf"),
      cns: get("cns"),
      cpf_ou_cns: get("cpf_ou_cns"),
      doc_tipo: get("doc_tipo"),
      mother_name: get("mother_name"),
      birth_date: get("birth_date"),
      professional_name: get("professional_name"),
      dt_inicial_atendimento: get("dt_inicial_atendimento"),
      dum: get("dum"),
      cbo: get("cbo"),
      cid: get("cid"),
      ciap: get("ciap"),
      qtd_filhos: get("qtd_filhos"),
    };
  });
}

function getValidDumEntries(moduleId, limit = 0) {
  if (!moduleId) return [];
  const dbInstance = getDb();
  let sql = `
      SELECT co_fat_cidadao, cpf_ou_cns, dum
      FROM module_items
      WHERE module_id = ?
        AND dum IS NOT NULL
        AND TRIM(dum) <> ''
      ORDER BY updated_at DESC
    `;
  const params = [moduleId];
  if (Number.isFinite(limit) && limit > 0) {
    sql += " LIMIT ?";
    params.push(limit);
  }
  const rows = dbInstance.prepare(sql).all(...params);
  return rows.map((row) => ({
    co_fat_cidadao: row.co_fat_cidadao,
    cpf_ou_cns: row.cpf_ou_cns,
    dum: row.dum,
  }));
}

function clearModuleItems(moduleId) {
  const dbInstance = getDb();
  const info = dbInstance
    .prepare("DELETE FROM module_items WHERE module_id = ?")
    .run(moduleId);
  dbInstance
    .prepare("DELETE FROM module_logs WHERE module_id = ?")
    .run(moduleId);
  dbInstance
    .prepare("DELETE FROM module_syncs WHERE module_id = ?")
    .run(moduleId);
  return { deleted: info.changes || 0 };
}

module.exports = {
  getLastItemId,
  hasMissingNames,
  getMissingNameIds,
  getMissingDtIds,
  persistModuleItems,
  markQueued,
  setSyncResponse,
  getSyncStats,
  runSql,
  listModuleItems,
  getPendingItems,
  getValidDumEntries,
  logModuleEvent,
  getModuleLogSummary,
  getModuleLogs,
  clearModuleItems,
  logSyncSessionStart,
  updateSyncSession,
  getLogMessages,
  dbPath,
};
