const fs = require('fs');
const path = require('path');
const { loadModulesConfig, getModuleConfig } = require('./config');
const { getSyncStats } = require('./storage');

const modulesDir = path.join(__dirname, '../../modules');

function getModuleManifest(moduleId) {
  const manifestPath = path.join(modulesDir, moduleId, 'module.json');
  if (!fs.existsSync(manifestPath)) return null;
  const raw = fs.readFileSync(manifestPath, 'utf-8');
  return JSON.parse(raw);
}

async function listModules() {
  if (!fs.existsSync(modulesDir)) {
    return [];
  }

  const modulesConfig = loadModulesConfig();
  const entries = fs
    .readdirSync(modulesDir, { withFileTypes: true })
    .filter((dirent) => dirent.isDirectory());

  const results = [];

  for (const entry of entries) {
    const manifest = getModuleManifest(entry.name);
    if (!manifest) continue;

    const settings = modulesConfig[manifest.id] || {};
    const required = manifest.requiredSettings || [];
    const missingSettings = required
      .filter((s) => !s.optional && !settings[s.key])
      .map((s) => s.key);

    let syncStats = null;
    try {
      syncStats = getSyncStats(manifest.id);
    } catch (err) {
      // Ambiente (ex: Node x Electron) pode não ter compilado o native module; ignora estatística.
      syncStats = null;
    }

    const blockedReason = missingSettings.length > 0 ? 'missing_required_settings' : null;

    results.push({
      id: manifest.id,
      name: manifest.name,
      version: manifest.version,
      blockedReason,
      missingSettings,
      syncStats
    });
  }

  return results;
}

async function runModule(moduleId, params = {}) {
  const modulePath = path.join(modulesDir, moduleId, 'index.js');
  if (!fs.existsSync(modulePath)) {
    throw new Error(`Módulo ${moduleId} não encontrado em ${modulePath}`);
  }
  // eslint-disable-next-line import/no-dynamic-require, global-require
  const mod = require(modulePath);
  if (!mod.run || typeof mod.run !== 'function') {
    throw new Error(`Módulo ${moduleId} não exporta função run`);
  }
  const moduleConfig = getModuleConfig(moduleId);
  return mod.run(params, moduleConfig);
}

module.exports = {
  listModules,
  getModuleManifest,
  runModule
};
