const fs = require('fs');
const path = require('path');
const { resolveConfigDirRoot } = require('./paths');

function resolveConfigDir() {
  if (process.env.EXTRATOR_CONFIG_DIR) {
    return path.resolve(process.env.EXTRATOR_CONFIG_DIR);
  }
  try {
    // eslint-disable-next-line global-require
    const { app } = require('electron');
    const electronApp = app || (require('electron').remote?.app);
    if (electronApp?.getPath) {
      return path.join(electronApp.getPath('userData'), 'config');
    }
  } catch (err) {
    // ignore when electron not available (e.g., CLI)
  }

  const legacyDir = path.join(process.cwd(), 'config');
  if (fs.existsSync(path.join(legacyDir, 'modules-config.json'))) {
    return legacyDir;
  }

  return resolveConfigDirRoot();
}

const configDir = resolveConfigDir();
const configPath = path.join(configDir, 'modules-config.json');
const GLOBAL_CONFIG_MODULE_ID = '__global__';

function ensureConfigFile() {
  if (!fs.existsSync(configPath)) {
    fs.mkdirSync(path.dirname(configPath), { recursive: true });
    fs.writeFileSync(configPath, JSON.stringify({}, null, 2));
  }
}

function loadModulesConfig() {
  ensureConfigFile();
  const raw = fs.readFileSync(configPath, 'utf-8');
  return JSON.parse(raw || '{}');
}

function saveModulesConfig(config) {
  ensureConfigFile();
  fs.writeFileSync(configPath, JSON.stringify(config, null, 2));
  return config;
}

function getModuleConfig(moduleId) {
  const cfg = loadModulesConfig();
  return cfg[moduleId] || {};
}

function updateModuleConfig(moduleId, updates) {
  const cfg = loadModulesConfig();
  cfg[moduleId] = { ...(cfg[moduleId] || {}), ...updates };
  saveModulesConfig(cfg);
  return cfg[moduleId];
}

module.exports = {
  loadModulesConfig,
  saveModulesConfig,
  getModuleConfig,
  updateModuleConfig,
  GLOBAL_CONFIG_MODULE_ID,
  configDir
};
