#!/usr/bin/env node
const yargs = require('yargs');
const { hideBin } = require('yargs/helpers');
const path = require('path');
const fs = require('fs');
require('dotenv').config({ path: path.join(__dirname, '../../.env') });
const { listModules, getModuleManifest, runModule } = require('../core/modules');
const { checkCoreUpdates, autoUpdateCore } = require('../core/updates');
const {
  runSql,
  dbPath,
  listModuleItems,
  clearModuleItems,
  getModuleLogSummary,
  getModuleLogs,
  getLogMessages
} = require('../core/storage');
const { getModuleConfig, updateModuleConfig } = require('../core/config');
const { resolveRootDir } = require('../core/paths');
const { sendPendingItems } = require('../../modules/filhos_de_minas/sender');
const TERMS_ACCEPT_GUIDANCE =
  'Leia e aceite os Termos de Uso do módulo (modules/filhos_de_minas/TERMS.md) e marque "termsAccepted=true" no config antes de executar (ex.: extrator-esus config-set --module filhos_de_minas --set termsAccepted=true).';
const FILHOS_MODULE_ID = 'filhos_de_minas';
const LOG_ENTRY_PREVIEW = 160;

async function handleListModules() {
  const modules = await listModules();
  if (modules.length === 0) {
    console.log('Nenhum módulo encontrado em modules/.');
    return;
  }
  modules.forEach((m) => {
    const status = m.blockedReason ? `bloqueado (${m.blockedReason})` : 'ok';
    console.log(`- ${m.name} (${m.id}) v${m.version} => ${status}`);
  });
}

async function handleRun(argv) {
  const moduleId = argv.module;
  if (!moduleId) {
    console.error('Use --module=<id> para selecionar o módulo.');
    process.exit(1);
  }
  const manifest = getModuleManifest(moduleId);
  if (!manifest) {
    console.error(`Módulo ${moduleId} não encontrado.`);
    process.exit(1);
  }
  const moduleConfig = getModuleConfig(moduleId);
  if (!moduleConfig.termsAccepted) {
    console.error(`Termos de uso do módulo Filhos de Minas não aceitos. ${TERMS_ACCEPT_GUIDANCE}`);
    process.exit(1);
  }
  const params = {
    page: argv.page || 1,
    pageSize: argv.pageSize || 100,
    filter: argv.filter || ''
  };

  console.log(`Executando módulo ${manifest.name} (v${manifest.version})...`);
  try {
    const result = await runModule(moduleId, params);
    const inserted = result?.persisted?.inserted || 0;
    const updated = result?.persisted?.updated || 0;
    console.log(`Sincronização concluída. Inseridos: ${inserted}, Atualizados: ${updated}`);
    if (result?.summary) {
      console.log('Resumo:', result.summary);
    }
  } catch (err) {
    console.error(`Erro ao executar módulo: ${err.message}`);
    process.exit(1);
  }
}

async function handleCheckUpdates() {
  const core = await checkCoreUpdates();
  console.log('Core:', core.status);
  const modules = await listModules();
  console.log('Módulos instalados (sem verificação de atualizações):');
  modules.forEach((m) => {
    const status = m.blockedReason ? `bloqueado (${m.blockedReason})` : 'ok';
    console.log(`- ${m.name} (${m.id}) v${m.version} => ${status}`);
  });
}

async function handleUpdateCore(argv) {
  try {
    const result = await autoUpdateCore({
      downloadDir: argv.downloadDir,
      apply: argv.apply
    });

    if (result.status === 'unreachable') {
      console.error('Serviço de atualização indisponível:', result.error || 'erro desconhecido');
      process.exit(1);
    }

    if (!result.downloaded) {
      console.log('Nenhuma atualização disponível ou já atualizado.');
      return;
    }

    if (result.applied) {
      console.log(`Atualização ${result.version} aplicada a partir de ${result.filePath}.`);
      console.log('A aplicação pode reiniciar ou solicitar fechamento para concluir a instalação.');
    } else {
      console.log(
        `Atualização ${result.version} baixada em ${result.filePath}. Abra o instalador para concluir.`
      );
    }
  } catch (err) {
    console.error('Falha ao atualizar o core:', err.message);
    process.exit(1);
  }
}

function handleSqlite(argv) {
  const sql = argv.sql;
  if (!sql) {
    console.error('Use --sql "<consulta>" para executar no SQLite local.');
    process.exit(1);
  }
  try {
    const rows = runSql(sql);
    console.log(`DB: ${dbPath}`);
    console.table(rows);
  } catch (err) {
    console.error('Erro ao executar SQL:', err.message);
    process.exit(1);
  }
}

function handleLocalList(argv) {
  const moduleId = argv.module;
  if (!moduleId) {
    console.error('Use --module=<id> para selecionar o módulo.');
    process.exit(1);
  }
  try {
    const res = listModuleItems({
      moduleId,
      filter: argv.filter || '',
      queued: argv.queued || 'all',
      status: argv.status || 'all',
      page: argv.page || 1,
      pageSize: argv.pageSize || 25
    });
    console.log(`DB: ${dbPath}`);
    console.log(
      `Página ${res.page} (tam=${res.pageSize}) — total ${res.total} registros | Gestantes únicas: ${res.summary.gestantes} | Média de filhos: ${res.summary.mediaFilhos} | CIDs únicos: ${res.summary.uniqueCids}`
    );
    console.table(res.rows);
  } catch (err) {
    console.error('Erro ao listar itens locais:', err.message);
    process.exit(1);
  }
}

function handleLocalClear(argv) {
  const moduleId = argv.module;
  if (!moduleId) {
    console.error('Use --module=<id> para selecionar o módulo.');
    process.exit(1);
  }
  try {
    const res = clearModuleItems(moduleId);
    console.log(`Banco local limpo para ${moduleId}. Registros removidos: ${res.deleted}`);
  } catch (err) {
    console.error('Erro ao limpar itens locais:', err.message);
    process.exit(1);
  }
}

function printLogSummary(summary) {
  if (!summary || summary.length === 0) {
    console.log('Nenhuma tentativa registrada ainda.');
    return;
  }
  console.log('Últimas tentativas:');
  console.table(
    summary.map((row) => ({
      attemptId: row.attempt_id,
      startedAt: row.started_at,
      total: row.total,
      success: row.success_count,
      errors: row.error_count
    }))
  );
}

function buildLogTable(entries = []) {
  return entries.map((entry) => {
    let details = entry.details || '';
    if (details.length > LOG_ENTRY_PREVIEW) {
      details = `${details.slice(0, LOG_ENTRY_PREVIEW)}…`;
    }
    return {
      id: entry.id,
      attemptId: entry.attempt_id,
      action: entry.action,
      status: entry.status,
      message: entry.message,
      details,
      createdAt: entry.created_at
    };
  });
}

function getMissingRequiredSettings(manifest, config) {
  if (!manifest?.requiredSettings) return [];
  return manifest.requiredSettings
    .filter((setting) => {
      if (setting.optional) return false;
      const value = config?.[setting.key];
      return value === undefined || value === null || value === '';
    })
    .map((setting) => setting.key);
}

function isSensitiveSetting(manifest, key) {
  const setting = manifest?.requiredSettings?.find((entry) => entry.key === key);
  if (setting?.type === 'secret') return true;
  if (setting?.sensitive) return true;
  return /password|senha|token|secret/i.test(key);
}

function maskConfigValue(value) {
  if (value === undefined || value === null || value === '') return value;
  const text = String(value);
  if (text.length <= 2) return '*'.repeat(text.length);
  const maskedLength = Math.min(Math.max(text.length - 2, 3), 6);
  return `${text[0]}${'*'.repeat(maskedLength)}${text.slice(-1)}`;
}

function handleLogs(argv) {
  const moduleId = argv.module;
  if (!moduleId) {
    console.error('Use --module=<id> para exibir logs.');
    process.exit(1);
  }
  const manifest = getModuleManifest(moduleId);
  if (!manifest) {
    console.error(`Módulo ${moduleId} não encontrado.`);
    process.exit(1);
  }
  const summary = getModuleLogSummary(moduleId, argv.limitAttempts || 6);
  printLogSummary(summary);
  const entries = getModuleLogs({
    moduleId,
    limit: argv.limitEntries || 100,
    attemptId: argv.attemptId,
    status: argv.status || 'all',
    search: argv.search || '',
    action: argv.action || null,
    messageTerm: argv.message || ''
  });
  if (!entries.length) {
    console.log('Nenhuma entrada de log encontrada com os filtros fornecidos.');
    return;
  }
  console.log(`Exibindo ${entries.length} registros de log:`);
  console.table(buildLogTable(entries));
}

function handleLogMessages(argv) {
  const moduleId = argv.module;
  if (!moduleId) {
    console.error('Use --module=<id> para exibir mensagens de log.');
    process.exit(1);
  }
  const action = argv.action;
  if (!action) {
    console.error('Informe --action=<nome> para filtrar mensagens por ação.');
    process.exit(1);
  }
  const manifest = getModuleManifest(moduleId);
  if (!manifest) {
    console.error(`Módulo ${moduleId} não encontrado.`);
    process.exit(1);
  }
  const messages = getLogMessages({
    moduleId,
    action,
    attemptId: argv.attemptId
  });
  if (!messages.length) {
    console.log('Nenhuma mensagem registrada para essa ação.');
    return;
  }
  console.log(`Mensagens encontradas para "${action}":`);
  messages.forEach((msg, idx) => {
    console.log(`${idx + 1}. ${msg}`);
  });
}

async function handleFilhosSend(argv) {
  if (argv.module && argv.module !== FILHOS_MODULE_ID) {
    console.error(`O comando send-filhos foi criado para o módulo ${FILHOS_MODULE_ID}.`);
    process.exit(1);
  }
  const moduleId = FILHOS_MODULE_ID;
  const manifest = getModuleManifest(moduleId);
  if (!manifest) {
    console.error(`Módulo ${moduleId} não encontrado.`);
    process.exit(1);
  }
  const config = getModuleConfig(moduleId);
  const missingSettings = getMissingRequiredSettings(manifest, config);
  if (missingSettings.length) {
    console.error(
      `Configuração incompleta para ${manifest.name}. Preencha ${missingSettings.join(
        ', '
      )} antes de enviar. Use config-get/config-set para ajustar.`
    );
    process.exit(1);
  }
  if (!config?.termsAccepted) {
    console.error(`Termos de uso do módulo Filhos de Minas não aceitos. ${TERMS_ACCEPT_GUIDANCE}`);
    process.exit(1);
  }
  try {
    const result = await sendPendingItems(config, { limit: argv.limit });
    console.log(
      `Envio concluído (attempt ${result.attemptId}): ${result.sent}/${result.total} registros. Erros: ${result.errors}.`
    );
    if (result.fatalError) {
      console.error('Erro fatal durante o envio:', result.fatalError.message || result.fatalError);
    }
  } catch (err) {
    console.error('Falha ao enviar Filhos de Minas:', err.message || err);
    if (err.curl) {
      console.log('Curl que falhou:', err.curl);
    }
    process.exit(1);
  }
}

function handleUninstall() {
  const rootDir = resolveRootDir();
  if (!fs.existsSync(rootDir)) {
    console.log('Nenhum dado local do Extrator foi encontrado para remover.');
    return;
  }
  try {
    fs.rmSync(rootDir, { recursive: true, force: true });
    console.log(`Dados locais removidos em ${rootDir}.`);
    console.log('Execute `npm uninstall -g extrator-pec-esus` se quiser remover o binário também.');
  } catch (err) {
    console.error('Falha ao remover os dados locais:', err.message);
    process.exit(1);
  }
}

function handleHelp() {
  const manualPath = path.join(__dirname, '../../README_CLI.md');
  console.log('Extrator PEC e-SUS — CLI');
  console.log('Comandos principais:');
  console.log('- list-modules');
  console.log('- check-updates');
  console.log('- update-core [--download-dir <dir>] [--apply true|false]');
  console.log('- run --module <id> [--page 1] [--pageSize 100] [--filter ""]');
  console.log('- local-list --module <id> [--filter ""] [--queued all|queued|pending] [--status all|success|error|none] [--page 1] [--pageSize 25]');
  console.log('- local-clear --module <id>');
  console.log('- config-get --module <id>');
  console.log('- config-set --module <id> [--set key=valor ...] [--json caminho.json]');
  console.log('- send-filhos [--module filhos_de_minas] [--limit <n>]');
  console.log('- uninstall (remove dados e downloads locais)');
  console.log('- logs --module <id> [--limitAttempts 6] [--limitEntries 100] [--status all|success|error] [--action <nome>] [--search <texto>] [--message <texto>] [--attemptId <id>]');
  console.log('- log-messages --module <id> --action <nome> [--attemptId <id>]');
  console.log('- sqlite --sql "<consulta>"');
  console.log('');
  if (fs.existsSync(manualPath)) {
    console.log(`Manual completo e exemplos: ${manualPath}`);
  } else {
    console.log('Manual completo: README_CLI.md (incluído no pacote).');
  }
  console.log('Use "--help" ao lado de um comando para detalhes (ex.: extrator-esus run --help).');
}

function parseKeyValues(list = []) {
  const updates = {};
  list.forEach((entry) => {
    if (!entry || typeof entry !== 'string') return;
    const idx = entry.indexOf('=');
    if (idx <= 0) return;
    const key = entry.slice(0, idx).trim();
    const val = entry.slice(idx + 1);
    if (key) updates[key] = val;
  });
  return updates;
}

function handleConfigGet(argv) {
  const moduleId = argv.module;
  if (!moduleId) {
    console.error('Use --module=<id> para selecionar o módulo.');
    process.exit(1);
  }
  const manifest = getModuleManifest(moduleId);
  if (!manifest) {
    console.error(`Módulo ${moduleId} não encontrado.`);
    process.exit(1);
  }
  const current = getModuleConfig(moduleId);
  const required = manifest.requiredSettings || [];
  const missing = required
    .filter((s) => !s.optional && !current[s.key])
    .map((s) => s.key);

  console.log(`Config do módulo ${manifest.name} (${moduleId}):`);
  console.table(
    Object.entries(current).map(([k, v]) => ({
      chave: k,
      valor: isSensitiveSetting(manifest, k) ? maskConfigValue(v) : v
    }))
  );
  if (required.length) {
    console.log('Configurações obrigatórias:');
    required.forEach((s) => {
      const status = missing.includes(s.key) ? 'FALTANDO' : 'ok';
      console.log(`- ${s.key} (${s.title || s.description || ''}) => ${status}`);
    });
  } else {
    console.log('Nenhuma configuração obrigatória declarada no manifest.');
  }
}

function handleConfigSet(argv) {
  const moduleId = argv.module;
  if (!moduleId) {
    console.error('Use --module=<id> para selecionar o módulo.');
    process.exit(1);
  }

  const manifest = getModuleManifest(moduleId);
  if (!manifest) {
    console.error(`Módulo ${moduleId} não encontrado.`);
    process.exit(1);
  }

  let updates = parseKeyValues(argv.set || []);
  if (argv.json) {
    try {
      const loaded = require(argv.json);
      if (typeof loaded === 'object' && loaded !== null) {
        updates = { ...updates, ...loaded };
      }
    } catch (err) {
      console.error(`Erro ao ler JSON ${argv.json}: ${err.message}`);
      process.exit(1);
    }
  }

  if (!Object.keys(updates).length) {
    console.error('Informe pelo menos uma configuração com --set key=valor ou --json caminho.json');
    process.exit(1);
  }

  const saved = updateModuleConfig(moduleId, updates);
  console.log(`Configurações salvas para ${moduleId}.`);
  console.table(
    Object.entries(saved).map(([k, v]) => ({
      chave: k,
      valor: v
    }))
  );
}

yargs(hideBin(process.argv))
  .command(
    'list-modules',
    'Lista módulos instalados',
    () => {},
    () => handleListModules()
  )
  .command(
    'run',
    'Executa um módulo (sincroniza e persiste no banco local)',
    (y) =>
      y
        .option('module', { alias: 'm', type: 'string', describe: 'ID do módulo', demandOption: true })
        .option('page', { type: 'number', default: 1, describe: 'Página inicial para o módulo' })
        .option('pageSize', {
          type: 'number',
          default: 100,
          describe: 'Tamanho da página para sincronização'
        })
        .option('filter', { type: 'string', default: '', describe: 'Filtro aplicado na execução' }),
    (argv) => handleRun(argv)
  )
  .command('check-updates', 'Verifica atualização do core', () => {}, () =>
    handleCheckUpdates()
  )
  .command(
    'update-core',
    'Baixa e aplica a atualização do core automaticamente para o SO atual',
    (y) =>
      y
        .option('download-dir', {
          type: 'string',
          describe: 'Diretório para salvar o instalador'
        })
        .option('apply', {
          type: 'boolean',
          default: true,
          describe: 'Executa o instalador assim que o download finalizar'
        }),
    (argv) => handleUpdateCore(argv)
  )
  .command(
    'sqlite',
    'Executa uma consulta no banco local (better-sqlite3)',
    (y) => y.option('sql', { type: 'string', describe: 'Consulta SQL a executar', demandOption: true }),
    (argv) => handleSqlite(argv)
  )
  .command(
    'local-list',
    'Lista registros locais do módulo (mesmo que a UI abre ao clicar no módulo)',
    (y) =>
      y
        .option('module', { alias: 'm', type: 'string', describe: 'ID do módulo', demandOption: true })
        .option('filter', { type: 'string', default: '', describe: 'Filtro por nome/CPF/CNS/CID/CIAP' })
        .option('queued', {
          type: 'string',
          choices: ['all', 'queued', 'pending'],
          default: 'all',
          describe: 'Filtrar por enviados/pedentes'
        })
        .option('status', {
          type: 'string',
          choices: ['all', 'success', 'error', 'none'],
          default: 'all',
          describe: 'Filtrar por status de envio'
        })
        .option('page', { type: 'number', default: 1, describe: 'Página' })
        .option('pageSize', { type: 'number', default: 25, describe: 'Registros por página' }),
    (argv) => handleLocalList(argv)
  )
  .command(
    'local-clear',
    'Limpa todos os registros locais de um módulo',
    (y) => y.option('module', { alias: 'm', type: 'string', demandOption: true, describe: 'ID do módulo' }),
    (argv) => handleLocalClear(argv)
  )
  .command(
    'config-get',
    'Mostra configurações do módulo e quais obrigatórias estão faltando',
    (y) => y.option('module', { alias: 'm', type: 'string', demandOption: true, describe: 'ID do módulo' }),
    (argv) => handleConfigGet(argv)
  )
  .command(
    'config-set',
    'Atualiza configurações do módulo via key=valor ou JSON',
    (y) =>
      y
        .option('module', { alias: 'm', type: 'string', demandOption: true, describe: 'ID do módulo' })
        .option('set', {
          type: 'array',
          describe: 'Parâmetros key=valor (use múltiplos --set para várias chaves)'
        })
        .option('json', {
          type: 'string',
          describe: 'Arquivo JSON com pares chave:valor'
        }),
    (argv) => handleConfigSet(argv)
  )
  .command(
    'send-filhos',
    'Envia os registros pendentes ao Filhos de Minas (após rodar `run`)',
    (y) =>
      y
        .option('module', {
          type: 'string',
          describe: 'ID do módulo (apenas filhos_de_minas é suportado atualmente)'
        })
        .option('limit', {
          type: 'number',
          describe: 'Limita o número de registros pendentes a enviar (padrão: todos)'
        }),
    (argv) => handleFilhosSend(argv)
  )
  .command(
    'uninstall',
    'Remove dados locais, configurações e downloads do Extrator',
    () => {},
    () => handleUninstall()
  )
  .command(
    ['logs', 'module-logs'],
    'Exibe logs de sincronização de um módulo',
    (y) =>
      y
        .option('module', {
          alias: 'm',
          type: 'string',
          demandOption: true,
          describe: 'ID do módulo para exibir os logs'
        })
        .option('limitAttempts', {
          type: 'number',
          default: 6,
          describe: 'Quantidade de tentativas recentes para mostrar no resumo'
        })
        .option('limitEntries', {
          type: 'number',
          default: 100,
          describe: 'Quantidade máxima de entradas de log'
        })
        .option('attemptId', {
          type: 'string',
          describe: 'ID do lote (attempt_id) para filtrar as entradas'
        })
        .option('status', {
          type: 'string',
          default: 'all',
          describe: 'Filtrar por status (all/success/error)'
        })
        .option('action', {
          type: 'string',
          describe: 'Filtrar por ação (ex.: send.item, batch.error, etc.)'
        })
        .option('search', {
          type: 'string',
          describe: 'Busca livre em message ou details'
        })
        .option('message', {
          type: 'string',
          describe: 'Busca por mensagem exata'
        }),
    (argv) => handleLogs(argv)
  )
  .command(
    'log-messages',
    'Lista mensagens distintas para uma ação específica',
    (y) =>
      y
        .option('module', {
          alias: 'm',
          type: 'string',
          demandOption: true,
          describe: 'ID do módulo'
        })
        .option('action', {
          type: 'string',
          demandOption: true,
          describe: 'Ação para recuperar as mensagens (ex.: send.api_error)'
        })
        .option('attemptId', {
          type: 'string',
          describe: 'Limita as mensagens a um lote específico'
        }),
    (argv) => handleLogMessages(argv)
  )
  .command(
    ['help', '*'],
    'Mostra ajuda rápida e caminho do manual CLI',
    () => {},
    () => handleHelp()
  )
  .demandCommand(
    1,
    'Informe um comando: help | list-modules | run | check-updates | update-core | send-filhos | uninstall | logs | log-messages'
  )
  .help()
  .parse();
